package org.mobicents.mojo.sbb;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Iterator;
import java.util.List;

import org.apache.maven.archiver.MavenArchiveConfiguration;
import org.apache.maven.archiver.MavenArchiver;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.archiver.jar.JarArchiver;
import org.codehaus.plexus.archiver.jar.Manifest;
import org.codehaus.plexus.archiver.jar.ManifestException;
import org.codehaus.plexus.archiver.jar.Manifest.Attribute;
import org.codehaus.plexus.util.IOUtil;


public abstract class AbstractSbbPluginMojo extends AbstractMojo
{

    public static final String MANIFEST_URI = "META-INF/MANIFEST.MF";

    /**
     * The maven project.
     * 
     * @parameter expression="${project}"
     * @required
     * @readonly
     */
    protected MavenProject project;

    /**
     * The location of the manifest file to be used within the deployable unit.
     * 
     * @parameter expression="${basedir}/src/main/resources/META-INF/MANIFEST.MF"
     */
    private File manifestFile;

    /**
     * The directory for the generated packaging.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     */
    private String outputDirectory;

    /**
     * Directory containing the generated JAR.
     * 
     * @parameter expression="${project.build.directory}"
     * @required
     * @readonly
     */
    protected File targetDirectory;

    /**
     * The directory where the packaging is built.
     * 
     * @parameter expression="${project.build.directory}/${project.build.finalName}"
     * @required
     */
    //private File packagingDirectory;
    public abstract File getTargetDirectory();

    /**
     * 
     * 
     * @parameter default-value=""
     */
    private String classpathPrefix = "";

    /**
     * The maven archive configuration to use.
     * 
     * @parameter
     */
    protected MavenArchiveConfiguration archive = new MavenArchiveConfiguration();

    /**
     * 
     */
    public abstract void execute() throws MojoExecutionException;

    /**
     * 
     * @return
     */
    public MavenProject getProject()
    {
        return project;
    }

    /**
     * If no deployment descriptor filesnames are found, check for the existence of alternates before failing.
     * 
     * Subclasses are not required to override this method.
     * 
     * @return
     */
    public String[] getAlternateDeploymentDescriptorFilenames()
    {
        return null;
    }

    public String getOutputDirectory()
    {
        return outputDirectory;
    }
    public String getClasspathPrefix()
    {
        return this.classpathPrefix;
    }

//    public String getArchiveName()
//    {
//        return archiveName;
//    }

    protected void includeCustomManifestFile( JarArchiver anArchiver )
        throws MojoExecutionException, DependencyResolutionRequiredException
    {
        File customManifestFile = manifestFile;

        if ( !customManifestFile.exists() )
        {
            getLog().info( "Could not find manifest file: " + manifestFile + " - Generating one" );
            generateManifest( anArchiver );
        }
        else
        {
            getLog().info( "Including custom manifest file[" + customManifestFile + "]" );
            archive.setManifestFile( customManifestFile );
        }
    }

    private void generateManifest( JarArchiver anArchiver )
        throws MojoExecutionException, DependencyResolutionRequiredException
    {
        Manifest mf = null;

        File manifestDir = new File( getTargetDirectory(), "META-INF" );
        if ( !manifestDir.exists() )
        {
            manifestDir.mkdirs();
        }
        File manifestFile = new File( manifestDir.getAbsolutePath(), "MANIFEST.MF" );
        MavenArchiver ma = new MavenArchiver();
        ma.setArchiver( anArchiver );
        ma.setOutputFile( manifestFile );

        PrintWriter printWriter = null;
        try
        {
            mf = ma.getManifest( getProject(), archive.getManifest() );
            Attribute classpathAttrib = mf.getMainSection().getAttribute( "Class-Path" );
            if ( classpathAttrib == null )
            {
                getLog().info( "Class-Path Attribute to MANIFEST" );
                StringBuffer classpath = new StringBuffer();
                List artifacts = project.getRuntimeClasspathElements();
                
                for ( Iterator iter = artifacts.iterator(); iter.hasNext(); )
                {
                    File f = new File( (String) iter.next() );
                    if ( f.isFile() )
                    {
                        if ( classpath.length() > 0 )
                        {
                            classpath.append( " " );
                        }
                        if ( !classpathPrefix.equals("") )
                            classpath.append( classpathPrefix + "/" );
                            
                        classpath.append( f.getName() );
                    }
                }

                if ( classpath.length() > 0 )
                {
                    classpathAttrib = new Manifest.Attribute( "Class-Path", classpath.toString() );
                    mf.addConfiguredAttribute( classpathAttrib );
                }
            }
            
            getLog().info( "\n" + "Class-Path" + "\n" + classpathAttrib.getValue() + "\n" );

            printWriter = new PrintWriter( new FileWriter( manifestFile ) );
            mf.write( printWriter );
            
            anArchiver.addConfiguredManifest(mf);
            

        }
        catch ( ManifestException e )
        {
            throw new MojoExecutionException( "Error preparing the manifest: " + e.getMessage(), e );
        }
        catch ( DependencyResolutionRequiredException e )
        {
            throw new MojoExecutionException( "Error preparing the manifest: " + e.getMessage(), e );
        }
        catch ( IOException e )
        {
            throw new MojoExecutionException( "Error preparing the manifest: " + e.getMessage(), e );
        }
        finally
        {
            IOUtil.close( printWriter );
            this.manifestFile = manifestFile;
            
        }

    }
}
