/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.test.rest.client;

import com.carrotsearch.randomizedtesting.RandomizedTest;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLContext;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLContexts;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.Version;
import org.elasticsearch.client.support.Headers;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.io.PathUtils;
import org.elasticsearch.common.logging.ESLogger;
import org.elasticsearch.common.logging.Loggers;
import org.elasticsearch.common.network.NetworkAddress;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.test.rest.client.RestException;
import org.elasticsearch.test.rest.client.RestPath;
import org.elasticsearch.test.rest.client.RestResponse;
import org.elasticsearch.test.rest.client.http.HttpRequestBuilder;
import org.elasticsearch.test.rest.client.http.HttpResponse;
import org.elasticsearch.test.rest.spec.RestApi;
import org.elasticsearch.test.rest.spec.RestSpec;

public class RestClient
implements Closeable {
    public static final String PROTOCOL = "protocol";
    public static final String TRUSTSTORE_PATH = "truststore.path";
    public static final String TRUSTSTORE_PASSWORD = "truststore.password";
    private static final ESLogger logger = Loggers.getLogger(RestClient.class);
    private static final Set<String> ALWAYS_ACCEPTED_QUERY_STRING_PARAMS = Sets.newHashSet((Object[])new String[]{"pretty", "source", "filter_path"});
    private final String protocol;
    private final RestSpec restSpec;
    private final CloseableHttpClient httpClient;
    private final Headers headers;
    private final InetSocketAddress[] addresses;
    private final Version esVersion;

    public RestClient(RestSpec restSpec, Settings settings, InetSocketAddress[] addresses) throws IOException, RestException {
        assert (addresses.length > 0);
        this.restSpec = restSpec;
        this.headers = new Headers(settings);
        this.protocol = settings.get(PROTOCOL, "http");
        this.httpClient = this.createHttpClient(settings);
        this.addresses = addresses;
        this.esVersion = this.readAndCheckVersion();
        logger.info("REST client initialized {}, elasticsearch version: [{}]", new Object[]{addresses, this.esVersion});
    }

    private Version readAndCheckVersion() throws IOException, RestException {
        RestApi restApi = this.restApi("info");
        assert (restApi.getPaths().size() == 1);
        assert (restApi.getMethods().size() == 1);
        String version = null;
        for (InetSocketAddress address : this.addresses) {
            RestResponse restResponse = new RestResponse(this.httpRequestBuilder(address).path(restApi.getPaths().get(0)).method(restApi.getMethods().get(0)).execute());
            this.checkStatusCode(restResponse);
            Object latestVersion = restResponse.evaluate("version.number");
            if (latestVersion == null) {
                throw new RuntimeException("elasticsearch version not found in the response");
            }
            if (version == null) {
                version = latestVersion.toString();
                continue;
            }
            if (latestVersion.equals(version)) continue;
            throw new IllegalArgumentException("provided nodes addresses run different elasticsearch versions");
        }
        return Version.fromString(version);
    }

    public Version getEsVersion() {
        return this.esVersion;
    }

    public RestResponse callApi(String apiName, Map<String, String> params, String body, Map<String, String> headers) throws IOException, RestException {
        String ignoreString;
        ArrayList<Integer> ignores = new ArrayList<Integer>();
        HashMap requestParams = null;
        if (params != null && Strings.hasLength((String)(ignoreString = (String)(requestParams = Maps.newHashMap(params)).remove("ignore")))) {
            try {
                ignores.add(Integer.valueOf(ignoreString));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("ignore value should be a number, found [" + ignoreString + "] instead");
            }
        }
        HttpRequestBuilder httpRequestBuilder = this.callApiBuilder(apiName, requestParams, body);
        for (Map.Entry<String, String> header : headers.entrySet()) {
            httpRequestBuilder.addHeader(header.getKey(), header.getValue());
        }
        logger.debug("calling api [{}]", new Object[]{apiName});
        HttpResponse httpResponse = httpRequestBuilder.execute();
        if (!httpResponse.supportsBody()) {
            ignores.add(404);
        }
        RestResponse restResponse = new RestResponse(httpResponse);
        this.checkStatusCode(restResponse, ignores);
        return restResponse;
    }

    private void checkStatusCode(RestResponse restResponse, List<Integer> ignores) throws RestException {
        if (ignores.contains(restResponse.getStatusCode())) {
            if (logger.isDebugEnabled()) {
                logger.debug("ignored non ok status codes {} as requested", new Object[]{ignores});
            }
            return;
        }
        this.checkStatusCode(restResponse);
    }

    private void checkStatusCode(RestResponse restResponse) throws RestException {
        if (restResponse.isError()) {
            throw new RestException("non ok status code [" + restResponse.getStatusCode() + "] returned", restResponse);
        }
    }

    private HttpRequestBuilder callApiBuilder(String apiName, Map<String, String> params, String body) {
        boolean indexCreateApi = "create".equals(apiName);
        String api = indexCreateApi ? "index" : apiName;
        RestApi restApi = this.restApi(api);
        HttpRequestBuilder httpRequestBuilder = this.httpRequestBuilder();
        HashMap pathParts = Maps.newHashMap();
        if (params != null) {
            for (Map.Entry<String, String> entry : params.entrySet()) {
                if (restApi.getPathParts().contains(entry.getKey())) {
                    pathParts.put(entry.getKey(), entry.getValue());
                    continue;
                }
                if (restApi.getParams().contains(entry.getKey()) || ALWAYS_ACCEPTED_QUERY_STRING_PARAMS.contains(entry.getKey())) {
                    httpRequestBuilder.addParam(entry.getKey(), entry.getValue());
                    continue;
                }
                throw new IllegalArgumentException("param [" + entry.getKey() + "] not supported in [" + restApi.getName() + "] api");
            }
        }
        if (indexCreateApi) {
            httpRequestBuilder.addParam("op_type", "create");
        }
        List<String> supportedMethods = restApi.getSupportedMethods(pathParts.keySet());
        if (Strings.hasLength((String)body)) {
            if (!restApi.isBodySupported()) {
                throw new IllegalArgumentException("body is not supported by [" + restApi.getName() + "] api");
            }
            if (supportedMethods.contains("GET") && RandomizedTest.rarely()) {
                logger.debug("sending the request body as source param with GET method", new Object[0]);
                httpRequestBuilder.addParam("source", body).method("GET");
            } else {
                httpRequestBuilder.body(body).method((String)RandomizedTest.randomFrom(supportedMethods));
            }
        } else {
            if (restApi.isBodyRequired()) {
                throw new IllegalArgumentException("body is required by [" + restApi.getName() + "] api");
            }
            httpRequestBuilder.method((String)RandomizedTest.randomFrom(supportedMethods));
        }
        RestPath restPath = (RestPath)RandomizedTest.randomFrom((Object[])restApi.getFinalPaths(pathParts));
        return httpRequestBuilder.pathParts(restPath.getPathParts());
    }

    private RestApi restApi(String apiName) {
        RestApi restApi = this.restSpec.getApi(apiName);
        if (restApi == null) {
            throw new IllegalArgumentException("rest api [" + apiName + "] doesn't exist in the rest spec");
        }
        return restApi;
    }

    protected HttpRequestBuilder httpRequestBuilder(InetSocketAddress address) {
        return new HttpRequestBuilder(this.httpClient).addHeaders(this.headers).protocol(this.protocol).host(NetworkAddress.formatAddress((InetAddress)address.getAddress())).port(address.getPort());
    }

    protected HttpRequestBuilder httpRequestBuilder() {
        InetSocketAddress address = (InetSocketAddress)RandomizedTest.randomFrom((Object[])this.addresses);
        return this.httpRequestBuilder(address);
    }

    protected CloseableHttpClient createHttpClient(Settings settings) throws IOException {
        SSLConnectionSocketFactory sslsf;
        String keystorePath = settings.get(TRUSTSTORE_PATH);
        if (keystorePath != null) {
            String keystorePass = settings.get(TRUSTSTORE_PASSWORD);
            if (keystorePass == null) {
                throw new IllegalStateException("truststore.path is provided but not truststore.password");
            }
            Path path = PathUtils.get((String)keystorePath, (String[])new String[0]);
            if (!Files.exists(path, new LinkOption[0])) {
                throw new IllegalStateException("truststore.path is set but points to a non-existing file");
            }
            try {
                KeyStore keyStore = KeyStore.getInstance("jks");
                try (InputStream is = Files.newInputStream(path, new OpenOption[0]);){
                    keyStore.load(is, keystorePass.toCharArray());
                }
                SSLContext sslcontext = SSLContexts.custom().loadTrustMaterial(keyStore, null).build();
                sslsf = new SSLConnectionSocketFactory(sslcontext);
            }
            catch (KeyManagementException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
                throw new RuntimeException(e);
            }
        }
        sslsf = SSLConnectionSocketFactory.getSocketFactory();
        Registry socketFactoryRegistry = RegistryBuilder.create().register("http", (Object)PlainConnectionSocketFactory.getSocketFactory()).register("https", (Object)sslsf).build();
        return HttpClients.createMinimal((HttpClientConnectionManager)new PoolingHttpClientConnectionManager(socketFactoryRegistry, null, null, null, 15L, TimeUnit.SECONDS));
    }

    @Override
    public void close() {
        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{this.httpClient});
    }
}

