/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import com.google.common.base.Charsets;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.ImmutableMap;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.indexedscripts.delete.DeleteIndexedScriptRequest;
import org.elasticsearch.action.indexedscripts.get.GetIndexedScriptRequest;
import org.elasticsearch.action.indexedscripts.put.PutIndexedScriptRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.HasContextAndHeaders;
import org.elasticsearch.common.ParseField;
import org.elasticsearch.common.ParseFieldMatcher;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.query.TemplateQueryParser;
import org.elasticsearch.script.CompiledScript;
import org.elasticsearch.script.ExecutableScript;
import org.elasticsearch.script.Script;
import org.elasticsearch.script.ScriptContext;
import org.elasticsearch.script.ScriptContextRegistry;
import org.elasticsearch.script.ScriptEngineService;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.ScriptMetrics;
import org.elasticsearch.script.ScriptMode;
import org.elasticsearch.script.ScriptModes;
import org.elasticsearch.script.ScriptStats;
import org.elasticsearch.script.SearchScript;
import org.elasticsearch.script.Template;
import org.elasticsearch.search.internal.SearchContext;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;

public class ScriptService
extends AbstractComponent
implements Closeable {
    static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    public static final String DEFAULT_SCRIPTING_LANGUAGE_SETTING = "script.default_lang";
    public static final String SCRIPT_CACHE_SIZE_SETTING = "script.cache.max_size";
    public static final int SCRIPT_CACHE_SIZE_DEFAULT = 100;
    public static final String SCRIPT_CACHE_EXPIRE_SETTING = "script.cache.expire";
    public static final String SCRIPT_INDEX = ".scripts";
    public static final String DEFAULT_LANG = "groovy";
    public static final String SCRIPT_AUTO_RELOAD_ENABLED_SETTING = "script.auto_reload_enabled";
    private final String defaultLang;
    private final Set<ScriptEngineService> scriptEngines;
    private final ImmutableMap<String, ScriptEngineService> scriptEnginesByLang;
    private final ImmutableMap<String, ScriptEngineService> scriptEnginesByExt;
    private final ConcurrentMap<CacheKey, CompiledScript> staticCache = ConcurrentCollections.newConcurrentMap();
    private final Cache<CacheKey, CompiledScript> cache;
    private final Path scriptsDirectory;
    private final ScriptModes scriptModes;
    private final ScriptContextRegistry scriptContextRegistry;
    private final ParseFieldMatcher parseFieldMatcher;
    private Client client = null;
    private final ScriptMetrics scriptMetrics = new ScriptMetrics();
    @Deprecated
    public static final ParseField SCRIPT_LANG = new ParseField("lang", "script_lang");
    @Deprecated
    public static final ParseField SCRIPT_FILE = new ParseField("script_file", new String[0]);
    @Deprecated
    public static final ParseField SCRIPT_ID = new ParseField("script_id", new String[0]);
    @Deprecated
    public static final ParseField SCRIPT_INLINE = new ParseField("script", new String[0]);

    @Inject
    public ScriptService(Settings settings, Environment env, Set<ScriptEngineService> scriptEngines, ResourceWatcherService resourceWatcherService, ScriptContextRegistry scriptContextRegistry) throws IOException {
        super(settings);
        this.parseFieldMatcher = new ParseFieldMatcher(settings);
        if (Strings.hasLength(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING))) {
            throw new IllegalArgumentException("script.disable_dynamic is not a supported setting, replace with fine-grained script settings. \nDynamic scripts can be enabled for all languages and all operations by replacing `script.disable_dynamic: false` with `script.inline: on` and `script.indexed: on` in elasticsearch.yml");
        }
        this.scriptEngines = scriptEngines;
        this.scriptContextRegistry = scriptContextRegistry;
        int cacheMaxSize = settings.getAsInt(SCRIPT_CACHE_SIZE_SETTING, (Integer)100);
        TimeValue cacheExpire = settings.getAsTime(SCRIPT_CACHE_EXPIRE_SETTING, null);
        this.logger.debug("using script cache with max_size [{}], expire [{}]", cacheMaxSize, cacheExpire);
        this.defaultLang = settings.get(DEFAULT_SCRIPTING_LANGUAGE_SETTING, DEFAULT_LANG);
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder();
        if (cacheMaxSize >= 0) {
            cacheBuilder.maximumSize((long)cacheMaxSize);
        }
        if (cacheExpire != null) {
            cacheBuilder.expireAfterAccess(cacheExpire.nanos(), TimeUnit.NANOSECONDS);
        }
        this.cache = cacheBuilder.removalListener((RemovalListener)new ScriptCacheRemovalListener()).build();
        ImmutableMap.Builder enginesByLangBuilder = ImmutableMap.builder();
        ImmutableMap.Builder enginesByExtBuilder = ImmutableMap.builder();
        for (ScriptEngineService scriptEngine : scriptEngines) {
            for (String type : scriptEngine.types()) {
                enginesByLangBuilder.put((Object)type, (Object)scriptEngine);
            }
            for (String ext : scriptEngine.extensions()) {
                enginesByExtBuilder.put((Object)ext, (Object)scriptEngine);
            }
        }
        this.scriptEnginesByLang = enginesByLangBuilder.build();
        this.scriptEnginesByExt = enginesByExtBuilder.build();
        this.scriptModes = new ScriptModes((Map<String, ScriptEngineService>)this.scriptEnginesByLang, scriptContextRegistry, settings);
        this.scriptsDirectory = env.scriptsFile();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Using scripts directory [{}] ", this.scriptsDirectory);
        }
        FileWatcher fileWatcher = new FileWatcher(this.scriptsDirectory);
        fileWatcher.addListener(new ScriptChangesListener());
        if (settings.getAsBoolean(SCRIPT_AUTO_RELOAD_ENABLED_SETTING, (Boolean)true).booleanValue()) {
            resourceWatcherService.add(fileWatcher);
        } else {
            fileWatcher.init();
        }
    }

    @Inject(optional=true)
    public void setClient(Client client) {
        this.client = client;
    }

    @Override
    public void close() throws IOException {
        IOUtils.close(this.scriptEngines);
    }

    private ScriptEngineService getScriptEngineServiceForLang(String lang) {
        ScriptEngineService scriptEngineService = (ScriptEngineService)this.scriptEnginesByLang.get((Object)lang);
        if (scriptEngineService == null) {
            throw new IllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        return scriptEngineService;
    }

    private ScriptEngineService getScriptEngineServiceForFileExt(String fileExtension) {
        ScriptEngineService scriptEngineService = (ScriptEngineService)this.scriptEnginesByExt.get((Object)fileExtension);
        if (scriptEngineService == null) {
            throw new IllegalArgumentException("script file extension not supported [" + fileExtension + "]");
        }
        return scriptEngineService;
    }

    public CompiledScript compile(Script script, ScriptContext scriptContext, HasContextAndHeaders headersContext, Map<String, String> params) {
        boolean notSupported;
        ScriptEngineService scriptEngineService;
        if (script == null) {
            throw new IllegalArgumentException("The parameter script (Script) must not be null.");
        }
        if (scriptContext == null) {
            throw new IllegalArgumentException("The parameter scriptContext (ScriptContext) must not be null.");
        }
        String lang = script.getLang();
        if (lang == null) {
            lang = this.defaultLang;
        }
        if (!this.canExecuteScript(lang, scriptEngineService = this.getScriptEngineServiceForLang(lang), script.getType(), scriptContext)) {
            throw new ScriptException("scripts of type [" + (Object)((Object)script.getType()) + "], operation [" + scriptContext.getKey() + "] and lang [" + lang + "] are disabled");
        }
        boolean expression = "expression".equals(script.getLang());
        boolean bl = notSupported = scriptContext.getKey().equals(ScriptContext.Standard.UPDATE.getKey()) || scriptContext.getKey().equals(ScriptContext.Standard.MAPPING.getKey());
        if (expression && notSupported) {
            throw new ScriptException("scripts of type [" + (Object)((Object)script.getType()) + "], operation [" + scriptContext.getKey() + "] and lang [" + lang + "] are not supported");
        }
        return this.compileInternal(script, headersContext, params);
    }

    public CompiledScript compileInternal(Script script, HasContextAndHeaders context, Map<String, String> params) {
        CacheKey cacheKey;
        CompiledScript compiledScript;
        if (script == null) {
            throw new IllegalArgumentException("The parameter script (Script) must not be null.");
        }
        String lang = script.getLang() == null ? this.defaultLang : script.getLang();
        ScriptType type = script.getType();
        String name = script.getScript();
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Compiling lang: [{}] type: [{}] script: {}", new Object[]{lang, type, name});
        }
        ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(lang);
        if (type == ScriptType.FILE) {
            CacheKey cacheKey2 = new CacheKey(scriptEngineService, name, null, params);
            CompiledScript compiledScript2 = (CompiledScript)this.staticCache.get(cacheKey2);
            if (compiledScript2 == null) {
                throw new IllegalArgumentException("Unable to find on disk file script [" + name + "] using lang [" + lang + "]");
            }
            return compiledScript2;
        }
        String code = script.getScript();
        if (type == ScriptType.INDEXED) {
            IndexedScript indexedScript = new IndexedScript(lang, name);
            name = indexedScript.id;
            code = this.getScriptFromIndex(indexedScript.lang, indexedScript.id, context);
        }
        if ((compiledScript = (CompiledScript)this.cache.getIfPresent((Object)(cacheKey = new CacheKey(scriptEngineService, type == ScriptType.INLINE ? null : name, code, params)))) == null) {
            try {
                compiledScript = new CompiledScript(type, name, lang, scriptEngineService.compile(code, params));
            }
            catch (Exception exception) {
                throw new ScriptException("Failed to compile " + (Object)((Object)type) + " script [" + name + "] using lang [" + lang + "]", exception);
            }
            this.scriptMetrics.onCompilation();
            this.cache.put((Object)cacheKey, (Object)compiledScript);
        }
        return compiledScript;
    }

    public void queryScriptIndex(GetIndexedScriptRequest request, ActionListener<GetResponse> listener) {
        String scriptLang = this.validateScriptLanguage(request.scriptLang());
        GetRequest getRequest = new GetRequest(request, SCRIPT_INDEX).type(scriptLang).id(request.id()).version(request.version()).versionType(request.versionType()).preference("_local");
        this.client.get(getRequest, listener);
    }

    private String validateScriptLanguage(String scriptLang) {
        if (scriptLang == null) {
            scriptLang = this.defaultLang;
        } else if (!this.scriptEnginesByLang.containsKey((Object)scriptLang)) {
            throw new IllegalArgumentException("script_lang not supported [" + scriptLang + "]");
        }
        return scriptLang;
    }

    String getScriptFromIndex(String scriptLang, String id, HasContextAndHeaders context) {
        if (this.client == null) {
            throw new IllegalArgumentException("Got an indexed script with no Client registered.");
        }
        scriptLang = this.validateScriptLanguage(scriptLang);
        GetRequest getRequest = new GetRequest(SCRIPT_INDEX, scriptLang, id);
        getRequest.copyContextAndHeadersFrom(context);
        GetResponse responseFields = this.client.get(getRequest).actionGet();
        if (responseFields.isExists()) {
            return ScriptService.getScriptFromResponse(responseFields);
        }
        throw new IllegalArgumentException("Unable to find script [.scripts/" + scriptLang + "/" + id + "]");
    }

    private void validate(BytesReference scriptBytes, String scriptLang) {
        block8: {
            try {
                XContentParser parser = XContentFactory.xContent(scriptBytes).createParser(scriptBytes);
                parser.nextToken();
                Template template = TemplateQueryParser.parse(scriptLang, parser, this.parseFieldMatcher, "params", "script", "template");
                if (Strings.hasLength(template.getScript())) {
                    try {
                        ScriptEngineService scriptEngineService = this.getScriptEngineServiceForLang(scriptLang);
                        if (this.isAnyScriptContextEnabled(scriptLang, scriptEngineService, ScriptType.INDEXED)) {
                            Object compiled = scriptEngineService.compile(template.getScript(), Collections.emptyMap());
                            if (compiled == null) {
                                throw new IllegalArgumentException("Unable to parse [" + template.getScript() + "] lang [" + scriptLang + "] (ScriptService.compile returned null)");
                            }
                        } else {
                            this.logger.warn("skipping compile of script [{}], lang [{}] as all scripted operations are disabled for indexed scripts", template.getScript(), scriptLang);
                        }
                        break block8;
                    }
                    catch (Exception e) {
                        throw new IllegalArgumentException("Unable to parse [" + template.getScript() + "] lang [" + scriptLang + "]", e);
                    }
                }
                throw new IllegalArgumentException("Unable to find script in : " + scriptBytes.toUtf8());
            }
            catch (IOException e) {
                throw new IllegalArgumentException("failed to parse template script", e);
            }
        }
    }

    public void putScriptToIndex(PutIndexedScriptRequest request, ActionListener<IndexResponse> listener) {
        String scriptLang = this.validateScriptLanguage(request.scriptLang());
        this.validate(request.source(), scriptLang);
        IndexRequest indexRequest = ((IndexRequest)new IndexRequest(request).index(SCRIPT_INDEX)).type(scriptLang).id(request.id()).version(request.version()).versionType(request.versionType()).source(request.source()).opType(request.opType()).refresh(true);
        this.client.index(indexRequest, listener);
    }

    public void deleteScriptFromIndex(DeleteIndexedScriptRequest request, ActionListener<DeleteResponse> listener) {
        String scriptLang = this.validateScriptLanguage(request.scriptLang());
        DeleteRequest deleteRequest = ((DeleteRequest)new DeleteRequest(request).index(SCRIPT_INDEX)).type(scriptLang).id(request.id()).refresh(true).version(request.version()).versionType(request.versionType());
        this.client.delete(deleteRequest, listener);
    }

    public static String getScriptFromResponse(GetResponse responseFields) {
        Map<String, Object> source = responseFields.getSourceAsMap();
        if (source.containsKey("template")) {
            try {
                XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON);
                Object template = source.get("template");
                if (template instanceof Map) {
                    builder.map((Map)template);
                    return builder.string();
                }
                return template.toString();
            }
            catch (IOException | ClassCastException e) {
                throw new IllegalStateException("Unable to parse " + responseFields.getSourceAsString() + " as json", e);
            }
        }
        if (source.containsKey("script")) {
            return source.get("script").toString();
        }
        try {
            XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON);
            builder.map(responseFields.getSource());
            return builder.string();
        }
        catch (IOException | ClassCastException e) {
            throw new IllegalStateException("Unable to parse " + responseFields.getSourceAsString() + " as json", e);
        }
    }

    public ExecutableScript executable(Script script, ScriptContext scriptContext, HasContextAndHeaders headersContext, Map<String, String> params) {
        return this.executable(this.compile(script, scriptContext, headersContext, params), script.getParams());
    }

    public ExecutableScript executable(CompiledScript compiledScript, Map<String, Object> vars) {
        return this.getScriptEngineServiceForLang(compiledScript.lang()).executable(compiledScript, vars);
    }

    public SearchScript search(SearchLookup lookup, Script script, ScriptContext scriptContext, Map<String, String> params) {
        CompiledScript compiledScript = this.compile(script, scriptContext, SearchContext.current(), params);
        return this.getScriptEngineServiceForLang(compiledScript.lang()).search(compiledScript, lookup, script.getParams());
    }

    private boolean isAnyScriptContextEnabled(String lang, ScriptEngineService scriptEngineService, ScriptType scriptType) {
        for (ScriptContext scriptContext : this.scriptContextRegistry.scriptContexts()) {
            if (!this.canExecuteScript(lang, scriptEngineService, scriptType, scriptContext)) continue;
            return true;
        }
        return false;
    }

    private boolean canExecuteScript(String lang, ScriptEngineService scriptEngineService, ScriptType scriptType, ScriptContext scriptContext) {
        assert (lang != null);
        if (!this.scriptContextRegistry.isSupportedContext(scriptContext)) {
            throw new IllegalArgumentException("script context [" + scriptContext.getKey() + "] not supported");
        }
        ScriptMode mode = this.scriptModes.getScriptMode(lang, scriptType, scriptContext);
        switch (mode) {
            case ON: {
                return true;
            }
            case OFF: {
                return false;
            }
            case SANDBOX: {
                return scriptEngineService.sandboxed();
            }
        }
        throw new IllegalArgumentException("script mode [" + (Object)((Object)mode) + "] not supported");
    }

    public ScriptStats stats() {
        return this.scriptMetrics.stats();
    }

    private static class IndexedScript {
        private final String lang;
        private final String id;

        IndexedScript(String lang, String script) {
            this.lang = lang;
            String[] parts = script.split("/");
            if (parts.length == 1) {
                this.id = script;
            } else {
                if (parts.length != 3) {
                    throw new IllegalArgumentException("Illegal index script format [" + script + "] should be /lang/id");
                }
                if (!parts[1].equals(this.lang)) {
                    throw new IllegalStateException("Conflicting script language, found [" + parts[1] + "] expected + [" + this.lang + "]");
                }
                this.id = parts[2];
            }
        }
    }

    private static final class CacheKey {
        final String lang;
        final String name;
        final String code;
        final Map<String, String> params;

        private CacheKey(ScriptEngineService service, String name, String code, Map<String, String> params) {
            this.lang = service.types()[0];
            this.name = name;
            this.code = code;
            this.params = params;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            if (!this.lang.equals(cacheKey.lang)) {
                return false;
            }
            if (this.name != null ? !this.name.equals(cacheKey.name) : cacheKey.name != null) {
                return false;
            }
            if (this.code != null ? !this.code.equals(cacheKey.code) : cacheKey.code != null) {
                return false;
            }
            return this.params.equals(cacheKey.params);
        }

        public int hashCode() {
            int result = this.lang.hashCode();
            result = 31 * result + (this.name != null ? this.name.hashCode() : 0);
            result = 31 * result + (this.code != null ? this.code.hashCode() : 0);
            result = 31 * result + this.params.hashCode();
            return result;
        }
    }

    public static enum ScriptType {
        INLINE(0, "inline"),
        INDEXED(1, "id"),
        FILE(2, "file");

        private final int val;
        private final ParseField parseField;

        public static ScriptType readFrom(StreamInput in) throws IOException {
            int scriptTypeVal = in.readVInt();
            for (ScriptType type : ScriptType.values()) {
                if (type.val != scriptTypeVal) continue;
                return type;
            }
            throw new IllegalArgumentException("Unexpected value read for ScriptType got [" + scriptTypeVal + "] expected one of [" + ScriptType.INLINE.val + "," + ScriptType.FILE.val + "," + ScriptType.INDEXED.val + "]");
        }

        public static void writeTo(ScriptType scriptType, StreamOutput out) throws IOException {
            if (scriptType != null) {
                out.writeVInt(scriptType.val);
            } else {
                out.writeVInt(ScriptType.INLINE.val);
            }
        }

        private ScriptType(int val, String name) {
            this.val = val;
            this.parseField = new ParseField(name, new String[0]);
        }

        public ParseField getParseField() {
            return this.parseField;
        }

        public String toString() {
            return this.name().toLowerCase(Locale.ROOT);
        }
    }

    private class ScriptChangesListener
    extends FileChangesListener {
        private ScriptChangesListener() {
        }

        private Tuple<String, String> scriptNameExt(Path file) {
            Path scriptPath = ScriptService.this.scriptsDirectory.relativize(file);
            int extIndex = scriptPath.toString().lastIndexOf(46);
            if (extIndex != -1) {
                String ext = scriptPath.toString().substring(extIndex + 1);
                String scriptName = scriptPath.toString().substring(0, extIndex).replace(scriptPath.getFileSystem().getSeparator(), "_");
                return new Tuple<String, String>(scriptName, ext);
            }
            return null;
        }

        @Override
        public void onFileInit(Path file) {
            block19: {
                Tuple<String, String> scriptNameExt;
                if (ScriptService.this.logger.isTraceEnabled()) {
                    ScriptService.this.logger.trace("Loading script file : [{}]", file);
                }
                if ((scriptNameExt = this.scriptNameExt(file)) != null) {
                    ScriptEngineService engineService = ScriptService.this.getScriptEngineServiceForFileExt(scriptNameExt.v2());
                    if (engineService == null) {
                        ScriptService.this.logger.warn("no script engine found for [{}]", scriptNameExt.v2());
                    } else {
                        try {
                            if (ScriptService.this.isAnyScriptContextEnabled(engineService.types()[0], engineService, ScriptType.FILE)) {
                                ScriptService.this.logger.info("compiling script file [{}]", file.toAbsolutePath());
                                try (InputStreamReader reader = new InputStreamReader(Files.newInputStream(file, new OpenOption[0]), Charsets.UTF_8);){
                                    String script = Streams.copyToString(reader);
                                    CacheKey cacheKey = new CacheKey(engineService, scriptNameExt.v1(), null, Collections.emptyMap());
                                    ScriptService.this.staticCache.put(cacheKey, new CompiledScript(ScriptType.FILE, scriptNameExt.v1(), engineService.types()[0], engineService.compile(script, Collections.emptyMap())));
                                    ScriptService.this.scriptMetrics.onCompilation();
                                    break block19;
                                }
                            }
                            ScriptService.this.logger.warn("skipping compile of script file [{}] as all scripted operations are disabled for file scripts", file.toAbsolutePath());
                        }
                        catch (Throwable e) {
                            ScriptService.this.logger.warn("failed to load/compile script [{}]", e, scriptNameExt.v1());
                        }
                    }
                }
            }
        }

        @Override
        public void onFileCreated(Path file) {
            this.onFileInit(file);
        }

        @Override
        public void onFileDeleted(Path file) {
            Tuple<String, String> scriptNameExt = this.scriptNameExt(file);
            if (scriptNameExt != null) {
                ScriptEngineService engineService = ScriptService.this.getScriptEngineServiceForFileExt(scriptNameExt.v2());
                assert (engineService != null);
                ScriptService.this.logger.info("removing script file [{}]", file.toAbsolutePath());
                ScriptService.this.staticCache.remove(new CacheKey(engineService, scriptNameExt.v1(), null, Collections.emptyMap()));
            }
        }

        @Override
        public void onFileChanged(Path file) {
            this.onFileInit(file);
        }
    }

    private class ScriptCacheRemovalListener
    implements RemovalListener<CacheKey, CompiledScript> {
        private ScriptCacheRemovalListener() {
        }

        public void onRemoval(RemovalNotification<CacheKey, CompiledScript> notification) {
            if (ScriptService.this.logger.isDebugEnabled()) {
                ScriptService.this.logger.debug("notifying script services of script removal due to: [{}]", notification.getCause());
            }
            ScriptService.this.scriptMetrics.onCacheEviction();
            for (ScriptEngineService service : ScriptService.this.scriptEngines) {
                try {
                    service.scriptRemoved((CompiledScript)notification.getValue());
                }
                catch (Exception e) {
                    ScriptService.this.logger.warn("exception calling script removal listener for script service", e, new Object[0]);
                }
            }
        }
    }
}

