package jp.co.bizreach.jdynamo;

import com.amazonaws.services.cloudwatch.AmazonCloudWatchClient;
import com.amazonaws.services.dynamodbv2.AmazonDynamoDBClient;
import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.AttributeValueUpdate;
import com.amazonaws.services.dynamodbv2.model.BatchGetItemRequest;
import com.amazonaws.services.dynamodbv2.model.BatchGetItemResult;
import com.amazonaws.services.dynamodbv2.model.BatchWriteItemRequest;
import com.amazonaws.services.dynamodbv2.model.BatchWriteItemResult;
import com.amazonaws.services.dynamodbv2.model.ConditionalCheckFailedException;
import com.amazonaws.services.dynamodbv2.model.DeleteItemRequest;
import com.amazonaws.services.dynamodbv2.model.DeleteRequest;
import com.amazonaws.services.dynamodbv2.model.DescribeTableRequest;
import com.amazonaws.services.dynamodbv2.model.DescribeTableResult;
import com.amazonaws.services.dynamodbv2.model.GetItemRequest;
import com.amazonaws.services.dynamodbv2.model.GetItemResult;
import com.amazonaws.services.dynamodbv2.model.KeysAndAttributes;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughputDescription;
import com.amazonaws.services.dynamodbv2.model.PutItemRequest;
import com.amazonaws.services.dynamodbv2.model.PutRequest;
import com.amazonaws.services.dynamodbv2.model.QueryRequest;
import com.amazonaws.services.dynamodbv2.model.QueryResult;
import com.amazonaws.services.dynamodbv2.model.ReturnValue;
import com.amazonaws.services.dynamodbv2.model.ScanRequest;
import com.amazonaws.services.dynamodbv2.model.ScanResult;
import com.amazonaws.services.dynamodbv2.model.UpdateItemRequest;
import com.amazonaws.services.dynamodbv2.model.UpdateItemResult;
import com.amazonaws.services.dynamodbv2.model.WriteRequest;
import jp.co.bizreach.jdynamo.action.DynamoQuery;
import jp.co.bizreach.jdynamo.action.DynamoScan;
import jp.co.bizreach.jdynamo.action.DynamoUpdateChain;
import jp.co.bizreach.jdynamo.admin.DynamoAdminClient;
import jp.co.bizreach.jdynamo.core.DynamoAppEnvirionment;
import jp.co.bizreach.jdynamo.core.DynamoThroughputAdjuster;
import jp.co.bizreach.jdynamo.data.DynamoAttributeDefinition;
import jp.co.bizreach.jdynamo.data.DynamoIndex;
import jp.co.bizreach.jdynamo.data.DynamoKey;
import jp.co.bizreach.jdynamo.data.DynamoMetaTable;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeWithValue;
import jp.co.bizreach.jdynamo.util.DynamoAttributeUtil;
import jp.co.bizreach.jdynamo.util.DynamoCloudWatchClient;
import jp.co.bizreach.jdynamo.util.DynamoTableNameResolver;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.ConstructorUtils;
import org.apache.commons.collections4.ListUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 一般的な Dynamo 操作をするためのクライアントクラス
 * Created by iwami on 2016/06/30.
 */
@Slf4j
public class DynamoClient {

    private AmazonDynamoDBClient amazonDynamoClient;

    private DynamoAppEnvirionment appEnvironment;

    private DynamoAdminClient adminClient;

    /**
     * jdynamo の共通設定
     */
    @Getter
    private DynamoClientSetting clientSetting;

    @Getter
    private DynamoCloudWatchClient cloudWatchClient;

    /**
     * テーブル名解決クラス。環境に応じてテーブル名を変えたいときなどに使って下さい。
     * <pre><code>
     * [コード例]
     * client.setTableNameResolver(new DynamoTableNameResolver.TableNamePrefixResolver("staging_"));
     * </code></pre>
     */
    @Setter
    private DynamoTableNameResolver tableNameResolver;

    public class DynamoClientPrivate {

        public AmazonDynamoDBClient getRawDynamoClient() {
            return amazonDynamoClient;
        }

        public void executeUpdate(DynamoUpdateChain updateChain) {
            DynamoMetaTable table = updateChain.getTable();
            DynamoKey key = updateChain.getKey();
            UpdateItemRequest request = new UpdateItemRequest().withTableName(getRealTableName(table))
                    .withKey(createDynamoKey(table, key)).withReturnValues(ReturnValue.ALL_NEW);
            UpdateItemResult updateItemResult = amazonDynamoClient.updateItem(updateChain.makeUpdateItemRequest(request));
            log.debug(updateItemResult.toString());
        }

        public UpdateItemResult executeUpdate(DynamoUpdateChain updateChain, ReturnValue returnValue) {
            DynamoMetaTable table = updateChain.getTable();
            DynamoKey key = updateChain.getKey();
            UpdateItemRequest request = new UpdateItemRequest().withTableName(getRealTableName(table))
                    .withKey(createDynamoKey(table, key)).withReturnValues(returnValue);
            try {
                UpdateItemResult updateItemResult = amazonDynamoClient.updateItem(updateChain.makeUpdateItemRequest(request));
                return updateItemResult;
            } catch (ConditionalCheckFailedException e) {
                throw new DynamoConditionalCheckFailedException(e);
            }
        }

        public QueryResult rawQuery(QueryRequest queryRequest) {
            if (log.isDebugEnabled()) {
                log.debug("query start.");
            }
            QueryResult queryResult = amazonDynamoClient.query(queryRequest);
            if (log.isDebugEnabled()) {
                log.debug("query finished. count = " + queryResult.getCount() + ", consumedCapacity = " + queryResult
                        .getConsumedCapacity());
            }
            return queryResult;
        }

        public QueryResult rawQueryExponentialBackoff(QueryRequest queryRequest,
                                                      DynamoThroughputAdjuster throughputAdjuster) {
            if (log.isDebugEnabled()) {
                log.debug("query start.");
            }
            QueryResult queryResult = throughputAdjuster.query(queryRequest);
            if (log.isDebugEnabled()) {
                log.debug("query finished. count = " + queryResult.getCount() + ", consumedCapacity = " + queryResult
                        .getConsumedCapacity());
            }
            return queryResult;

        }

        public ScanResult rawScan(ScanRequest scanRequest, DynamoThroughputAdjuster throughputAdjuster) {
            return throughputAdjuster.scan(scanRequest);
        }

        public <T> List<T> toRecords(DynamoMetaTable<T> table, List<Map<String, AttributeValue>> items) {
            List<T> models = new ArrayList<>();
            for (Map<String, AttributeValue> itemMap : items) {
                models.add(toRecord(table, itemMap));
            }
            return models;
        }

        public <T> T toRecord(DynamoMetaTable<T> table, Map<String, AttributeValue> itemMap) {
            if (itemMap == null) {
                return null;
            }
            try {
                T record = (T) ConstructorUtils.invokeConstructor(table.getRecordClass(), new Object[0]);

                for (Map.Entry<String, AttributeValue> entry : itemMap.entrySet()) {
                    table.storeFieldByItem(record, entry, clientSetting);
                }

                return record;
            } catch (ReflectiveOperationException e) {
                log.error(e.getMessage(), e);
            }
            return null;
        }

        public <T> Map<String,AttributeValue> createKey(DynamoMetaTable<T> table, Object partitionKey, Object
                sortKey) {
            Map<String,AttributeValue> key = new HashMap<>();

            DynamoAttributeDefinition partitionKeyAttr = table.getPartitionKeyAttr();
            DynamoAttributeDefinition sortKeyAttr = table.getSortKeyAttr();

            key.put(partitionKeyAttr.getDynamoAttrName(),
                    DynamoAttributeUtil.createAttributeValue(partitionKeyAttr.getMappingType(), partitionKey));
            key.put(sortKeyAttr.getDynamoAttrName(),
                    DynamoAttributeUtil.createAttributeValue(sortKeyAttr.getMappingType(), sortKey));
            return key;
        }

        public String getRealTableName(DynamoMetaTable table) {
            return tableNameResolver.resolveTableName(table.getBaseTableName());
        }

        public ScanResult rawScan(ScanRequest scanRequest) {
            return amazonDynamoClient.scan(scanRequest);
        }

        public <T> ProvisionedThroughputDescription getTableProvisionedThroughput(DynamoMetaTable<T> table) {
            DescribeTableRequest request = new DescribeTableRequest().withTableName(getRealTableName(table));
            DescribeTableResult describeTableResult = amazonDynamoClient.describeTable(request);
            return describeTableResult.getTable().getProvisionedThroughput();
        }

    }

    private DynamoClientPrivate clientPrivate = new DynamoClientPrivate();

    /**
     * インスタンスを初期化します。
     * @param dynamoClient 素の AmazonDynamoDBClient
     */
    public DynamoClient(AmazonDynamoDBClient dynamoClient, DynamoAppEnvirionment env) {
        this.amazonDynamoClient = dynamoClient;
        this.appEnvironment = env;
        this.clientSetting = new DynamoClientSetting();
        tableNameResolver = new DynamoTableNameResolver.TableNamePrefixResolver("");

        log.info("DynamoClient created.");
    }

    public DescribeTableResult describeTable(DynamoMetaTable table) {
        return amazonDynamoClient.describeTable(clientPrivate.getRealTableName(table));
    }

    public void enableCloudWatchClient(AmazonCloudWatchClient dynamoCloudWatchClient) {
        cloudWatchClient = new DynamoCloudWatchClient(clientPrivate, dynamoCloudWatchClient);
    }

    public DynamoAdminClient useAdminClient() {
        if (appEnvironment != DynamoAppEnvirionment.LOCAL) {
            throw new IllegalStateException("useAdminClient() is only usable for LOCAL mode");
        }
        if (adminClient == null) {
            adminClient = new DynamoAdminClient(amazonDynamoClient, clientPrivate);
        }
        return adminClient;
    }

    /**
     * アイテムを1件PUTします。
     * @param table
     * @param record
     */
    public void put(DynamoMetaTable table, Object record) {
        PutItemRequest request = makePutItemRequest(table, record);
        if (log.isDebugEnabled()) {
            log.debug(request.toString());
        }
        amazonDynamoClient.putItem(request);
    }

    /**
     * アイテムを1件PUTします。
     * スループット例外が発生したとき、内部でリトライ処理をして確実にアイテムを返します。
     * @param table
     * @param record
     */
    public void putWithAdjustThroughput(DynamoMetaTable table, Object record) {
        PutItemRequest request = makePutItemRequest(table, record);
        if (log.isDebugEnabled()) {
            log.debug(request.toString());
        }
        new DynamoThroughputAdjuster(amazonDynamoClient).put(request);
    }

    /**
     * BatchWriteItem を発行します。（最大25件）
     * <i>スループットを超えたとき、エラーにはならずに一部のアイテムがPUTできないことがあります。その場合戻り値が1以上の値になります。</i>
     * @param table
     * @param records
     * @return 処理されなかったレコード件数
     */
    public int batchPut(DynamoMetaTable table, List<Object> records) {
        BatchWriteItemResult result = amazonDynamoClient.batchWriteItem(makeBatchPutItemRequest(table, records));
        if (result.getUnprocessedItems().isEmpty()) {
            return 0;
        }
        return result.getUnprocessedItems().entrySet().stream().mapToInt(entry -> entry.getValue().size())
                .sum();
    }

    /**
     * BatchWriteItem を発行します。（最大25件）
     * スループット例外が発生したとき、内部でリトライ処理をして確実にアイテムをPUTします。
     * @param table
     * @param records
     */
    public void batchPutWithAdjustThroughput(DynamoMetaTable table, List<Object> records) {
        BatchWriteItemResult result = amazonDynamoClient.batchWriteItem(makeBatchPutItemRequest(table, records));
        while (! result.getUnprocessedItems().isEmpty()) {
            log.info("found unprocessedItems. retry BatchWriteItemRequest.");
            result = innerBatchPutWithAdjustThroughput(table, result.getUnprocessedItems());
        }
    }

    /**
     * データを一括PUTします。25件を超える場合、25件ごとに分割してPUTリクエストを発行します。
     * スループット例外が発生したとき、内部でリトライ処理をして確実にアイテムをPUTします。そのため、処理が完了するまで長い時間が掛かることがあります。
     * @param table
     * @param records
     */
    public void batchPutWithSplit(DynamoMetaTable table, List<Object> records) {
        List<List<Object>> partition = ListUtils.partition(records, 25);
        for (List<Object> dynamoRecords : partition) {
            BatchWriteItemRequest rawRequest = makeBatchPutItemRequest(table, dynamoRecords);
            BatchWriteItemResult result = new DynamoThroughputAdjuster(amazonDynamoClient).batchPut(rawRequest);
            while (! result.getUnprocessedItems().isEmpty()) {
                int size = result.getUnprocessedItems().entrySet().stream().mapToInt(entry -> entry.getValue().size())
                        .sum();
                log.info("found unprocessedItems (" + size + "). retry BatchWriteItemRequest.");
                result = innerBatchPutWithAdjustThroughput(table, result.getUnprocessedItems());
            }

        }
    }


    /**
     * 指定された record のうち、null以外にセットされたフィールドのみ更新します。
     * <i>このメソッドは簡易的な更新処理にのみ使い、通常は updateChain を使って下さい。</i>
     * @param table
     * @param key
     * @param record
     * @see #updateChain(DynamoMetaTable, DynamoKey)
     */
    public void simpleUpdate(DynamoMetaTable table, DynamoKey key, Object record) {
        amazonDynamoClient.updateItem(makeUpdateItemRequest(table, key, record));
    }

    /**
     * アイテムを1件DELETEします。
     * @param table
     * @param key
     */
    public void delete(DynamoMetaTable table, DynamoKey key) {
        amazonDynamoClient.deleteItem(makeDeleteItemRequest(table, key, null));
    }

    /**
     * 条件付きDELETEを発行します。
     * 条件を満たさないとき、ConditionalCheckFailedException が発生します。
     * @param table
     * @param key
     * @param condition
     * @see com.amazonaws.services.dynamodbv2.model.ConditionalCheckFailedException
     */
    public void delete(DynamoMetaTable table, DynamoKey key, DynamoAttributeWithValue condition) {
        amazonDynamoClient.deleteItem(makeDeleteItemRequest(table, key, condition));
    }

    /**
     * BatchWriteItem でアイテムを一括で削除します。
     * <i>スループットを超えたとき、エラーにはならずに一部のアイテムがDELETEできないことがあります。その場合戻り値が1以上の値になります。</i>
     * @param table
     * @param keys
     * @return 処理されなかったレコード件数
     */
    public int batchDelete(DynamoMetaTable table, List<DynamoKey> keys) {
        BatchWriteItemResult result = amazonDynamoClient.batchWriteItem(makeBatchDeleteRequest(table, keys));
        if (result.getUnprocessedItems().isEmpty()) {
            return 0;
        }
        Collection<List<WriteRequest>> values = result.getUnprocessedItems().values();
        int count = 0;
        for (List<WriteRequest> list : values) {
            count += list.size();
        }
        return count;
    }

    /**
     * 指定したテーブルで UpdateItem チェーンを発行します。
     * <i>このメソッドを呼び出すだけではコマンドは発行されないので注意してください。</i>
     * @param table
     * @param key
     * @return
     * @see DynamoUpdateChain
     */
    public DynamoUpdateChain updateChain(DynamoMetaTable table, DynamoKey key) {
        return new DynamoUpdateChain(clientPrivate, table, key);
    }

    public <T> DynamoUpdateChain updateChain(DynamoMetaTable<T> table, T rec) {
        DynamoKey key = table.createKey(rec);
        return updateChain(table, key);
    }

/*
    public void update(DynamoMetaTable table, DynamoKey key, Consumer<DynamoUpdateChain> consumer) {
        DynamoUpdateChain chain = new DynamoUpdateChain(clientPrivate, table, key);
        consumer.accept(chain);
        chain.execute();
    }
*/

    public <T> T get(DynamoMetaTable<T> table, Object partitionKey, Object sortKey) {
        GetItemRequest request = new GetItemRequest()
                .withTableName(clientPrivate.getRealTableName(table))
                .withKey(clientPrivate.createKey(table, partitionKey, sortKey));
        GetItemResult itemResult = amazonDynamoClient.getItem(request);
        return clientPrivate.toRecord(table, itemResult.getItem());
    }

    public <T> T get(DynamoMetaTable<T> table, Object partitionKey) {
        GetItemRequest request = new GetItemRequest()
                .withTableName(clientPrivate.getRealTableName(table))
                .withKey(createKey(table, partitionKey));
        GetItemResult itemResult = amazonDynamoClient.getItem(request);
        return clientPrivate.toRecord(table, itemResult.getItem());
    }

    /**
     * BatchGetItem によりアイテムを取得します（最大100件）。
     * IMPORTANT: スループットが超過した場合、ProvisionedThroughputExceededException が発生することがあります。
     * また、スループットがギリギリの場合、例外が発生しなくてもアイテムを全て取得できない可能性があります。
     * 確実に取得したい場合は、batchGetWithAdjustThroughput を使ってください。
     * @param table
     * @param keys
     * @param <T>
     * @return
     */
    public <T> List<T> batchGet(DynamoMetaTable<T> table, List<DynamoKey> keys) {
        List<Map<String, AttributeValue>> dynamoKeys = new ArrayList<>();
        for (DynamoKey key : keys) {
            dynamoKeys.add(clientPrivate.createKey(table, key.getPartitionKey(), key.getSortKey()));
        }
        BatchGetItemRequest request = new BatchGetItemRequest()
                .addRequestItemsEntry(clientPrivate.getRealTableName(table), new KeysAndAttributes().withKeys(dynamoKeys));
        BatchGetItemResult itemResult = amazonDynamoClient.batchGetItem(request);
        return clientPrivate.toRecords(table, itemResult.getResponses().get(clientPrivate.getRealTableName(table)));
    }

    /**
     * BatchGetItem を発行します。リトライを繰り返し、確実にアイテムを取得します（最大100件）。
     * @param table
     * @param keys
     * @param <T>
     * @return
     */
    public <T> List<T> batchGetWithAdjustThroughput(DynamoMetaTable<T> table, List<DynamoKey> keys) {
        List<Map<String, AttributeValue>> dynamoKeys = new ArrayList<>();
        for (DynamoKey key : keys) {
            dynamoKeys.add(clientPrivate.createKey(table, key.getPartitionKey(), key.getSortKey()));
        }
        BatchGetItemRequest request = new BatchGetItemRequest()
                .addRequestItemsEntry(clientPrivate.getRealTableName(table), new KeysAndAttributes().withKeys(dynamoKeys));
        BatchGetItemResult itemResult = new DynamoThroughputAdjuster(amazonDynamoClient).batchGet(request);

        List<T> results = new ArrayList<T>();
        results.addAll(clientPrivate.toRecords(table, itemResult.getResponses().get(clientPrivate.getRealTableName(table))));

        while (! itemResult.getUnprocessedKeys().isEmpty()) {
            List<Map<String, AttributeValue>> unprocessedKeys = new ArrayList<>();
            Collection<KeysAndAttributes> values = itemResult.getUnprocessedKeys().values();
            for (KeysAndAttributes value : values) {
                unprocessedKeys.addAll(value.getKeys());
            }
            log.info("found unprocessedKeys [size = " + unprocessedKeys.size() + "]. retry batchGet");
            itemResult = innerBatchGetWithAdjustThroughput(table, unprocessedKeys);
            results.addAll(clientPrivate.toRecords(table, itemResult.getResponses().get(clientPrivate.getRealTableName(table))));
        }
        return results;
    }

    public <T> List<T> batchGetMany(DynamoMetaTable<T> table, List<DynamoKey> keys) {
        throw new UnsupportedOperationException("this method is not implemented.");
    }

    /**
     * 指定したテーブルで QUERY チェーンを発行します（メインインデックス）。
     * <i>このメソッドを呼び出すだけではコマンドは発行されないので注意してください。</i>
     * @param table
     * @param <T>
     * @return
     */
    public <T> DynamoQuery<T> query(DynamoMetaTable<T> table) {
        return new DynamoQuery<>(clientPrivate, table);
    }

    /**
     * 指定したテーブル・指定したインデックスで QUERY チェーンを発行します。
     * <i>このメソッドを呼び出すだけではコマンドは発行されないので注意してください。</i>
     * @param table
     * @param index
     * @param <T>
     * @return
     */
    public <T> DynamoQuery<T> query(DynamoMetaTable<T> table, DynamoIndex index) {
        return new DynamoQuery<>(clientPrivate, table, index);
    }

    /**
     * 指定したテーブルで SCAN チェーンを発行します（メインインデックス）。
     * <i>このメソッドを呼び出すだけではコマンドは発行されないので注意してください。</i>
     * @param table
     * @param <T>
     * @return
     */
    public <T> DynamoScan<T> scan(DynamoMetaTable<T> table) {
        return new DynamoScan<>(clientPrivate, table);
    }

    /**
     * 指定したテーブルの予約済スループット数を返します。
     * @param table
     * @param <T>
     * @return
     */
    public <T> ProvisionedThroughputDescription getProvisionedThroughput(DynamoMetaTable<T> table) {
        return clientPrivate.getTableProvisionedThroughput(table);
    }

    //--------------------------------------------------------------- Private Methods

    private <T> Map<String,AttributeValue> createKey(DynamoMetaTable<T> table, Object partitionKey) {
        Map<String,AttributeValue> key = new HashMap<>();
        DynamoAttributeDefinition partitionKeyAttr = table.getPartitionKeyAttr();
        key.put(partitionKeyAttr.getDynamoAttrName(),
                DynamoAttributeUtil.createAttributeValue(partitionKeyAttr.getMappingType(), partitionKey));
        return key;
    }

    private <T> Map<String,AttributeValue> createDynamoKey(DynamoMetaTable<T> table, DynamoKey dynamoKey) {
        Map<String,AttributeValue> key = new HashMap<>();

        DynamoAttributeDefinition partitionKeyAttr = table.getPartitionKeyAttr();
        DynamoAttributeDefinition sortKeyAttr = table.getSortKeyAttr();

        key.put(partitionKeyAttr.getDynamoAttrName(),
                DynamoAttributeUtil.createAttributeValue(partitionKeyAttr.getMappingType(), dynamoKey.getPartitionKey()));
        if (sortKeyAttr != null) {
            key.put(sortKeyAttr.getDynamoAttrName(),
                    DynamoAttributeUtil.createAttributeValue(sortKeyAttr.getMappingType(), dynamoKey.getSortKey()));
        }
        return key;
    }

    private PutItemRequest makePutItemRequest(DynamoMetaTable table, Object record) {
        Map<String, AttributeValue> item = createItem(table, record);
        PutItemRequest request = new PutItemRequest().withTableName(clientPrivate.getRealTableName(table))
                .withItem(item);
        if (log.isDebugEnabled()) {
            log.debug("new PutItemRequest().\nitem = " + item);
        }
        return request;
    }


    private UpdateItemRequest makeUpdateItemRequest(DynamoMetaTable table, DynamoKey key, Object record) {
        return new UpdateItemRequest().withTableName(clientPrivate.getRealTableName(table))
                .withKey(createDynamoKey(table, key)).withAttributeUpdates(createUpdateItem(table, record));
    }

/*
    private DeleteItemRequest makeDeleteItemRequest(DynamoMetaTable table, DynamoKey key) {
        return new DeleteItemRequest().withTableName(clientPrivate.getRealTableName(table))
                .withKey(createDynamoKey(table, key));
    }
*/

    private DeleteItemRequest makeDeleteItemRequest(DynamoMetaTable table, DynamoKey key, DynamoAttributeWithValue
            condition) {

        DeleteItemRequest request = new DeleteItemRequest();
        request.withTableName(clientPrivate.getRealTableName(table))
                .withKey(createDynamoKey(table, key));
        if (condition != null) {
            List<Integer> indexes = Arrays.asList(new Integer[]{1});
            String expression = condition.getExpression(indexes);
            request.withConditionExpression(expression);

            Map<String, String> names = new HashMap<>();
            condition.appendAttributeNames(names, 1);

            Map<String, AttributeValue> values = new HashMap<>();
            condition.appendAttributeValues(values, 1);

            request.withExpressionAttributeNames(names);
            request.withExpressionAttributeValues(values);
        }
        return request;
    }

    private Map<String,AttributeValue> createItem(DynamoMetaTable table, Object record) {
        return table.createItem(record);
    }

    private Map<String,AttributeValueUpdate> createUpdateItem(DynamoMetaTable table, Object record) {
        return table.createUpdateItem(record);
    }

    private BatchWriteItemResult innerBatchPutWithAdjustThroughput(
            DynamoMetaTable table, Map<String, List<WriteRequest>> unprocessedItems) {

        List<WriteRequest> writeRequests = new ArrayList<>();
        for (List<WriteRequest> list : unprocessedItems.values()) {
            writeRequests.addAll(list);
        }
        BatchWriteItemRequest rawRequest = new BatchWriteItemRequest().addRequestItemsEntry(clientPrivate
                .getRealTableName(table), writeRequests);

        BatchWriteItemResult result = new DynamoThroughputAdjuster(amazonDynamoClient).batchPut(rawRequest);
        return result;
    }

    private BatchWriteItemRequest makeBatchDeleteRequest(DynamoMetaTable table, List<DynamoKey> keys) {
        List<WriteRequest> requests = new ArrayList<>();
        for (DynamoKey key : keys) {
            requests.add(new WriteRequest(new DeleteRequest(createDynamoKey(table, key))));
        }
        return new BatchWriteItemRequest().addRequestItemsEntry(clientPrivate.getRealTableName(table), requests);
    }

    private BatchWriteItemRequest makeBatchPutItemRequest(DynamoMetaTable table, List<Object> records) {
        List<WriteRequest> requests = new ArrayList<>();
        for (Object record : records) {
            requests.add(new WriteRequest(new PutRequest(createItem(table, record))));
        }
        return new BatchWriteItemRequest().addRequestItemsEntry(clientPrivate.getRealTableName(table), requests);
    }

    private <T> BatchGetItemResult innerBatchGetWithAdjustThroughput(
            DynamoMetaTable<T> table, List<Map<String, AttributeValue>> unprocessedKeys) {
        BatchGetItemRequest request = new BatchGetItemRequest()
                .addRequestItemsEntry(clientPrivate.getRealTableName(table), new KeysAndAttributes().withKeys(unprocessedKeys));
        BatchGetItemResult itemResult = new DynamoThroughputAdjuster(amazonDynamoClient).batchGet(request);
        return itemResult;
    }

}
