package jp.co.bizreach.jdynamo.action;

import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.QueryRequest;
import com.amazonaws.services.dynamodbv2.model.QueryResult;
import com.amazonaws.services.dynamodbv2.model.ReturnConsumedCapacity;
import com.amazonaws.services.dynamodbv2.model.Select;
import jp.co.bizreach.jdynamo.DynamoClient;
import jp.co.bizreach.jdynamo.core.DynamoThroughputAdjuster;
import jp.co.bizreach.jdynamo.data.DynamoIndex;
import jp.co.bizreach.jdynamo.data.DynamoMappingAttributeType;
import jp.co.bizreach.jdynamo.data.DynamoMetaTable;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeWithValue;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeSupport;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Query Action のラッパークラスです。
 * <pre><code>
 *     [コード例]
 *     client.query(MessageTable.as()).key(101L, 10L).getAsList();
 * </code></pre>
 * クラスはチェーン形式になっています。必ず最後に getAsList() もしくは getAsListDesc() を呼び出してください。
 * Created by iwami on 2016/06/30.
 */
@Slf4j
public class DynamoQuery<T> extends DynamoBaseAction<T> {

    private static final String KNAME = "#kname";

    private List<DynamoQueryValue> keyValues = new ArrayList<>();

    private DynamoThroughputAdjuster throughputAdjuster;

    private QueryResult lastQueryResult;

    private boolean enableLoggingInfo = false;

    /**
     * ライブラリが内部的に呼び出します。
     * <i>クライアントコードからは呼び出さないでください</i>
     * @param client
     * @param table
     */
    public DynamoQuery(DynamoClient.DynamoClientPrivate client, DynamoMetaTable table) {
        super(client, table);
    }

    /**
     * ライブラリが内部的に呼び出します。
     * <i>クライアントコードからは呼び出さないでください</i>
     * @param client
     * @param table
     * @param index
     */
    public DynamoQuery(DynamoClient.DynamoClientPrivate client, DynamoMetaTable table, DynamoIndex index) {
        super(client, table, index);
    }

    /**
     * パーティションキーを指定します。
     * @param partitionKey
     * @return
     */
    public DynamoQuery<T> key(Object partitionKey) {
        keyValues.add(new DynamoQueryValue(index.getPartitionAttributeDefinition(), partitionKey));
        return this;
    }

    /**
     * パーティションキーおよびソートキーを指定します。
     * <i>多くの場合、これらのキーが事前にわかっている場合は Query ではなく Get を使ったほうが効率が良いです</i>
     * @param partitionKey
     * @param sortKey
     * @return
     */
    public DynamoQuery<T> key(Object partitionKey, Object sortKey) {
        if (isTargetIndexPartition()) {
            throw new IllegalStateException("can't use sort-key. Because current context use partition-key index (not"
                    + " define sort-key)");
        }

        keyValues.add(new DynamoQueryValue(index.getPartitionAttributeDefinition(), partitionKey));
        keyValues.add(new DynamoQueryValue(index.getSortAttributeDefinition(), sortKey));
        return this;
    }

    /**
     * パーティションキーおよびソートキー（範囲指定）を指定します。
     * @param partitionKey
     * @param sortKeyFrom
     * @param sortKeyTo
     * @return
     */
    public DynamoQuery<T> keyRange(Object partitionKey, Object sortKeyFrom, Object sortKeyTo) {
        keyValues.add(new DynamoQueryValue(index.getPartitionAttributeDefinition(), partitionKey));
        keyValues.add(new DynamoQueryValue(index.getSortAttributeDefinition(), new RangeValue(sortKeyFrom, sortKeyTo)));
        return this;
    }

    /**
     * フィルタ条件を追加します。
     * このメソッドを複数回呼び出した場合、それらの条件はANDで繋がれます。
     * @param cond
     * @return
     */
    public DynamoQuery<T> filter(DynamoAttributeWithValue cond) {
        filterValues.add(new DynamoQueryValue(cond.getAttributeDefinition(), cond));
        return this;
    }

    /**
     * OR条件のフィルタ条件を追加します。
     * @param conditions
     * @return
     */
    public DynamoQuery<T> filterOr(DynamoAttributeWithValue... conditions) {
        filterValues.add(new DynamoFilterOr(Arrays.asList(conditions)));
        return this;
    }

    /**
     * 取得するアイテムの内容を特定の列だけに限定したいときに使います。
     * @param attrs
     * @return
     */
    public DynamoQuery<T> projection(DynamoAttributeSupport... attrs) {
        List<String> names = Arrays.stream(attrs).map(attr -> attr.getDynamoAttrName()).collect(Collectors.toList());
        projectionNames = names;
        return this;
    }

    /**
     * 取得するアイテムの最大数を制限したいときに使います。
     * @param limit
     * @return
     */
    public DynamoQuery<T> limit(Integer limit) {
        this.limit = limit;
        return this;
    }

    /**
     * クエリー実行時、スループットが超えないようにウェイトを入れます（Exponential Backoff Algorithm）。
     * これによってメソッドの実行時間が長くなる可能性がありますが、エラーが発生する確率が減ります。
     * @return
     */
    public DynamoQuery<T> withAdjustThroughput() {
        throughputAdjuster = new DynamoThroughputAdjuster(client.getRawDynamoClient());
        return this;
    }

    /**
     * クエリー実行時、スループットが超えないようにウェイトを入れます（Exponential Backoff Algorithm）。
     * これによってメソッドの実行時間が長くなる可能性がありますが、エラーが発生する確率が減ります。
     * @param retryLimitMillisecond 最大Nミリ秒ウェイトを入れます。実際には累乗でウェイトが入るため、ここの値よりも長くウェイトが入る可能性があります。
     * @return
     */
    public DynamoQuery<T> withAdjustThroughput(long retryLimitMillisecond) {
        throughputAdjuster = new DynamoThroughputAdjuster(client.getRawDynamoClient(), retryLimitMillisecond);
        return this;
    }

    /**
     * Query のリクエスト情報をINFOログに出力します。
     * <p>出力されるのは、実際にDynamoにリクエストを発行した後になります。</p>
     * @return
     */
    public DynamoQuery<T> withLoggingInfo() {
        enableLoggingInfo = true;
        return this;
    }

    /**
     * クエリを実行し、結果を返します。結果はソートキーの昇順になります。
     * <p>
     * 結果セットが大きい場合、1回で全ての結果を取得できない場合があります。
     * その場合、複数回呼び出すことで全ての結果を取得できます。
     * </p>
     * @return
     */
    public List<T> getAsList() {

        QueryRequest queryRequest = makeQueryRequest(true);
        QueryResult queryResult = innerExecuteQuery(queryRequest);

        lastQueryResult = queryResult;
        return client.toRecords(table, queryResult.getItems());
    }

    /**
     * クエリを実行し、結果を返します。結果はソートキーの降順になります。
     * <p>
     * 結果セットが大きい場合、1回で全ての結果を取得できない場合があります。
     * その場合、複数回呼び出すことで全ての結果を取得できます。
     * </p>
     * @return
     */
    public List<T> getAsListDesc() {
        QueryRequest queryRequest = makeQueryRequest(false);
        QueryResult queryResult = innerExecuteQuery(queryRequest);

        lastQueryResult = queryResult;
        return client.toRecords(table, queryResult.getItems());
    }

    /**
     * クエリ実行後、まだ結果セットが残っているかどうかを返します。
     * @return 結果セットが残っていれば true
     */
    public boolean isMoreRecords() {
        return lastQueryResult != null && lastQueryResult.getLastEvaluatedKey() != null;
    }

    /**
     * クエリを実行し、その件数のみを返します。
     * @return
     */
    public Integer getCount() {
        QueryResult queryResult = client.rawQuery(makeQueryRequest(true).withSelect(Select.COUNT));
        return queryResult.getCount();
    }


    //--------------------------------------------------------------- Private Methods

    private QueryRequest makeQueryRequest(boolean scanIndexForward) {

        String keyConditionExpression = makeKeyConditionExpression();

        String filterExpression = makeFilterExpression();

        Map<String, String> expressionAttributeNames = makeExpressionAttributeNames();
        Map<String, AttributeValue> expressionAttributeValues = makeExpressionAttributeValues();

        String projectionExpression = makeProjectionExpression();
        Select select = projectionExpression != null ? Select.SPECIFIC_ATTRIBUTES : Select.ALL_ATTRIBUTES;

        ReturnConsumedCapacity returnConsumedCapacity = ReturnConsumedCapacity.TOTAL;

        String realTableName = client.getRealTableName(table);
        if (log.isDebugEnabled()) {
            log.debug("new QueryRequest().\nrealTableName = " + realTableName + "\nkeyConditionExpression = "
                    + keyConditionExpression + "\nfilterExpression = "
                    + filterExpression + "\nexpressionAttributeNames = " + expressionAttributeNames + "\n"
                    + "expressionAttributeValues = " + expressionAttributeValues + "\nprojectionExpression = "
                    + projectionExpression);
        }

        return new QueryRequest()
                .withTableName(realTableName)
                .withIndexName(getIndexName())
                .withKeyConditionExpression(keyConditionExpression)
                .withFilterExpression(filterExpression)
                .withExpressionAttributeNames(expressionAttributeNames)
                .withExpressionAttributeValues(expressionAttributeValues)
                .withSelect(select)
                .withProjectionExpression(projectionExpression)
                .withReturnConsumedCapacity(returnConsumedCapacity)
                .withLimit(limit)
                .withExclusiveStartKey(lastQueryResult != null ? lastQueryResult.getLastEvaluatedKey() : null)
                .withScanIndexForward(scanIndexForward);
    }

    private String makeKeyConditionExpression() {
        StringBuilder sb = new StringBuilder();
        int idx = 1;

        // #kname1 = :kvalue1 AND #kname2 BETWEEN :kvalue2 AND :kvalue3

        for (DynamoQueryValue keyValue : keyValues) {
            if (sb.length() > 0) {
                sb.append(" AND ");
            }
            if (keyValue.getValue() instanceof RangeValue) {
                sb.append(KNAME + idx + " BETWEEN :kvalue" + idx + " AND :kvalue" + (idx + 1));
                ++idx;
            } else {
                sb.append(KNAME + idx + " = :kvalue" + idx);
            }
            ++idx;
        }

        return sb.toString();
    }

    private Map<String, String> makeExpressionAttributeNames() {
        Map<String, String> results = new HashMap<>();
        int idx = 1;
        for (DynamoQueryValue keyValue : keyValues) {
            results.put(KNAME + idx, keyValue.getAttributeDefinition().getDynamoAttrName());
            ++idx;
            if (idx > keyValues.size()) {
                break;
            }
        }

        idx = 1;
        for (DynamoCondExpression expression : filterValues) {
            expression.appendNames(Arrays.asList(idx), results);
            ++idx;
        }

        return results;
    }

    private Map<String, AttributeValue> makeExpressionAttributeValues() {
        Map<String, AttributeValue> results = new HashMap<>();
        int idx = 1;
        for (DynamoQueryValue keyValue : keyValues) {
            Object value = keyValue.getValue();
            if (value instanceof RangeValue) {
                Object subKeyValue = ((RangeValue)value).from;
                appendKeyValue(results, keyValue.getAttributeDefinition().getMappingType(), idx, subKeyValue);
                ++idx;

                subKeyValue = ((RangeValue)value).to;
                appendKeyValue(results, keyValue.getAttributeDefinition().getMappingType(), idx, subKeyValue);
            } else {
                appendKeyValue(results, keyValue.getAttributeDefinition().getMappingType(), idx, value);
            }
            ++idx;
        }

        idx = 1;

        for (DynamoCondExpression expression : filterValues) {
            Object filterValue = expression.getTargetValue();
            DynamoMappingAttributeType mappingType = expression.getMappingType();
            appendFilterValue(results, expression, Arrays.asList(idx));
/*
            if (filterValue instanceof RangeValue) {
                Object subFilterValue = ((RangeValue)filterValue).from;
                appendFilterValue(results, mappingType, idx, subFilterValue);
                ++idx;

                subFilterValue = ((RangeValue)filterValue).to;
                appendFilterValue(results, mappingType, idx, subFilterValue);
            } else {
                appendFilterValue(results, expression, Arrays.asList(idx));
            }
*/
            ++idx;
        }

        return results;
    }


    private QueryResult innerExecuteQuery(QueryRequest queryRequest) {
        if (enableLoggingInfo) {
            log.info(queryRequest.toString());
        }

        QueryResult queryResult;
        if (throughputAdjuster != null) {
            queryResult = client.rawQueryExponentialBackoff(queryRequest, throughputAdjuster);
        } else {
            queryResult = client.rawQuery(queryRequest);
        }
        return queryResult;
    }

}
