package jp.co.bizreach.jdynamo.action;

import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.ReturnConsumedCapacity;
import com.amazonaws.services.dynamodbv2.model.ScanRequest;
import com.amazonaws.services.dynamodbv2.model.Select;
import jp.co.bizreach.jdynamo.DynamoClient;
import jp.co.bizreach.jdynamo.core.DynamoThroughputAdjuster;
import jp.co.bizreach.jdynamo.data.DynamoMappingAttributeType;
import jp.co.bizreach.jdynamo.data.DynamoMetaTable;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeSupport;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeWithValue;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

/**
 * Scan Action のラッパークラスです。
 * <pre><code>
 *     [コード例]
 *     client.scan(MessageTable.as()).executeSync(5, (items, scanContext) -&gt; {
 *         // ここにスキャンされたアイテムが items に入っています（最大1MB）
 *         // この処理はスキャンが完了するまで複数回呼び出されます。
 *     });
 * </code></pre>
 * クラスはチェーン形式になっています。必ず最後に executeSync() を呼び出してください。
 * Created by iwami on 2016/07/01.
 */
@Slf4j
public class DynamoScan<T> extends DynamoBaseAction<T> {

    @Getter
    private DynamoThroughputAdjuster throughputAdjuster;

    public DynamoScan(DynamoClient.DynamoClientPrivate client, DynamoMetaTable table) {
        super(client, table);
    }

    @NoArgsConstructor
    public static class ScanContext {
        @Getter
        @Setter
        private Exception causeError;

        @Getter
        @Setter
        private ScanState state;

        public boolean isError() {
            return causeError != null;
        }
    }

    /**
     * フィルタ条件を指定します。
     * @param cond
     * @return
     */
    public DynamoScan<T> filter(DynamoAttributeWithValue cond) {
        filterValues.add(new DynamoQueryValue(cond.getAttributeDefinition(), cond));
        return this;
    }

    /**
     * Projection を指定します。
     * 指定しない場合、全属性が取得されます。
     * @param attrs
     * @return
     */
    public DynamoScan<T> projection(DynamoAttributeSupport... attrs) {
        List<String> names = Arrays.stream(attrs).map(attr -> attr.getDynamoAttrName()).collect(Collectors.toList());
        projectionNames = names;
        return this;
    }

    /**
     * スループット超過時のリトライ処理を有効にします。
     * スキャン処理はスループットが上がりやすいので、なるべく有効にしてください。
     * @return
     */
    public DynamoScan<T> withAdjustThroughput() {
        throughputAdjuster = new DynamoThroughputAdjuster(client.getRawDynamoClient());
        return this;
    }

    /**
     * 取得するレコードの最大数を制限したいときに使います。
     * <p>通常、スキャン処理を実行すると毎回1MB分までレコードを取得します。limit を指定すると、このとき指定した件数までしかレコードを取得しなくなります。</p>
     * @param limit
     * @return
     */
    public DynamoScan<T> limit(Integer limit) {
        this.limit = limit;
        return this;
    }

    /**
     * 並列Scan処理を実行します。
     * スキャンが完了するまで、メソッドはブロックされます。
     * @param threadCount 並列スレッド数
     * @param callbackFunction スキャン中に呼び出されるコールバック関数
     * @return
     */
    public DynamoScanResult executeSync(int threadCount, BiConsumer<List<T>, ScanContext> callbackFunction) {
        ExecutorService executor = new SafetyThreadPoolExecutor(threadCount, threadCount, 0L,
                TimeUnit.MILLISECONDS,
                new ArrayBlockingQueue<Runnable>(threadCount));

        DynamoScanResult result = new DynamoScanResult();
        long totalScanCount;
        try {
            List<Future<Long>> futures = new ArrayList<>();
            for (int segment = 0; segment < threadCount; segment++) {
                ScanContext scanContext = new ScanContext();
                result.addScanContext(scanContext);
                ScanActionRunner<T> scanRunner = new ScanActionRunner<>(
                        this, segment, threadCount, table, scanContext, client, callbackFunction);
                futures.add(executor.submit(scanRunner));
            }

            totalScanCount = getTotalScanCount(callbackFunction, futures);
        } finally {
            executor.shutdown();
        }

        log.info("scan executor [" + table.getBaseTableName() + "] shutdown. total count = " + totalScanCount);
        result.setScannedCount(totalScanCount);

        return result;
    }

    //--------------------------------------------------------------- Private Methods

    private long getTotalScanCount(BiConsumer<List<T>, ScanContext> callbackFunction,
                                   List<Future<Long>> futures) {
        long totalScanCount = 0;
        for (Future<Long> future : futures) {
            try {
                Long result = future.get();
                if (result == null) {
                    continue;
                }
                log.info("scan result: [" + table.getBaseTableName() + "] count = " + result);
                totalScanCount += result;
            } catch (InterruptedException | ExecutionException e) {
                log.error(e.getMessage(), e);
                ScanContext scanContext = new ScanContext();
                if (e.getCause() instanceof Exception) {
                    scanContext.setCauseError((Exception) e.getCause());
                }
                callbackFunction.accept(null, scanContext);
            }
        }
        return totalScanCount;
    }

    /* package */ ScanRequest makeScanRequest(
            int segment, int totalSegment, Map<String, AttributeValue> exclusiveStartKey) {
        ScanRequest request = makeScanRequest();
        request.withSegment(segment).withTotalSegments(totalSegment);
        request.withLimit(limit);
        request.withExclusiveStartKey(exclusiveStartKey);
        return request;
    }

    private ScanRequest makeScanRequest() {

        String filterExpression = makeFilterExpression();
        Map<String, String> expressionAttributeNames = makeExpressionAttributeNames();
        Map<String, AttributeValue> expressionAttributeValues = makeExpressionAttributeValues();

        String projectionExpression = makeProjectionExpression();
        Select select = projectionExpression != null ? Select.SPECIFIC_ATTRIBUTES : Select.ALL_ATTRIBUTES;

        ReturnConsumedCapacity returnConsumedCapacity = ReturnConsumedCapacity.TOTAL;

        if (log.isDebugEnabled()) {
            log.debug("new ScanRequest().\nfilterExpression = "
                    + filterExpression + "\nexpressionAttributeNames = " + expressionAttributeNames + "\n"
                    + "expressionAttributeValues = " + expressionAttributeValues + "\nprojectionExpression = "
                    + projectionExpression);
        }

        return new ScanRequest()
                .withTableName(client.getRealTableName(table))
                .withIndexName(getIndexName())
                .withFilterExpression(filterExpression)
                .withExpressionAttributeNames(expressionAttributeNames)
                .withExpressionAttributeValues(expressionAttributeValues)
                .withSelect(select)
                .withProjectionExpression(projectionExpression)
                .withReturnConsumedCapacity(returnConsumedCapacity)
                .withLimit(limit);
    }

    private Map<String, String> makeExpressionAttributeNames() {
        Map<String, String> results = new HashMap<>();
        int idx = 1;
        for (DynamoCondExpression expression : filterValues) {
            String filterName = expression.getDynamoAttrName();
            if (results.containsValue(filterName) == false) {
                results.put("#fname" + idx, filterName);
                ++idx;
            }
        }
        return results.isEmpty() ? null : results;
    }

    private Map<String, AttributeValue> makeExpressionAttributeValues() {
        Map<String, AttributeValue> results = new HashMap<>();
        int idx = 1;

        for (DynamoCondExpression expression : filterValues) {
            Object filterValue = expression.getTargetValue();
            DynamoMappingAttributeType mappingType = expression.getMappingType();
            appendFilterValue(results, mappingType, idx, filterValue);
/*
            if (filterValue instanceof RangeValue) {
                Object subFilterValue = ((RangeValue)filterValue).from;
                appendFilterValue(results, mappingType, idx, subFilterValue);
                ++idx;

                subFilterValue = ((RangeValue)filterValue).to;
                appendFilterValue(results, mappingType, idx, subFilterValue);
            } else {
                appendFilterValue(results, mappingType, idx, filterValue);
            }
*/
            ++idx;
        }
        return results.isEmpty() ? null : results;
    }

}
