package jp.co.bizreach.jdynamo.action;

import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.ReturnValue;
import com.amazonaws.services.dynamodbv2.model.UpdateItemRequest;
import com.amazonaws.services.dynamodbv2.model.UpdateItemResult;
import jp.co.bizreach.jdynamo.DynamoClient;
import jp.co.bizreach.jdynamo.DynamoConditionalCheckFailedException;
import jp.co.bizreach.jdynamo.data.DynamoKey;
import jp.co.bizreach.jdynamo.data.DynamoMetaTable;
import jp.co.bizreach.jdynamo.data.DynamoUpdateValue;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeWithValue;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * UpdateItem Action のラッパークラスです。
 * <pre><code>
 *     [コード例]
 *     client.updateChain(table, new DynamoKey(101L, 10L))
 *           .set(table.text.set("updated"))
 *           .set(table.intCount.increment(4))
 *           .execute();
 * </code></pre>
 * クラスはチェーン形式になっています。必ず最後に execute() もしくは executeAndGet() を呼び出してください。
 * Created by iwami on 2016/07/04.
 */
@Slf4j
public class DynamoUpdateChain<T> {

    private DynamoClient.DynamoClientPrivate client;

    @Getter
    private DynamoMetaTable<T> table;
    @Getter
    private DynamoKey key;

    private List<DynamoUpdateValue> values = new ArrayList<>();

    private DynamoAttributeWithValue condition;

    public DynamoUpdateChain(DynamoClient.DynamoClientPrivate client, DynamoMetaTable table, DynamoKey key) {
        this.client = client;
        this.table = table;
        this.key = key;
    }

    /**
     * 更新する値を指定します。
     * @param value
     * @return
     */
    public DynamoUpdateChain set(DynamoUpdateValue value) {
        values.add(value);
        return this;
    }

    /**
     * UPDATE 処理を実行します。
     */
    public void execute() {
        client.executeUpdate(this);
    }

    /**
     * UPDATE 処理を実行し、UPDATE後のレコードの内容を返します。
     * @return UPDATE後のレコード（今回更新しなかった値も含めて、全属性が入っている）
     */
    public T executeAndGet() {
        UpdateItemResult updateItemResult = client.executeUpdate(this, ReturnValue.ALL_NEW);
        Map<String, AttributeValue> attributes = updateItemResult.getAttributes();
        if (log.isDebugEnabled()) {
            log.debug(attributes.toString());
        }
        return client.toRecord(table, attributes);
    }

    /**
     * UPDATE 処理を実行し、UPDATE後のレコードの内容を返します。
     * 条件付き更新に失敗しても例外が発生せず、null が返ります。
     * @return UPDATE後のレコード（今回更新しなかった値も含めて、全属性が入っている）。条件付き更新に失敗したら null
     */
    public T executeConditionalAndGet() {
        try {
            UpdateItemResult updateItemResult = client.executeUpdate(this, ReturnValue.ALL_NEW);
            Map<String, AttributeValue> attributes = updateItemResult.getAttributes();
            if (log.isDebugEnabled()) {
                log.debug(attributes.toString());
            }
            return client.toRecord(table, attributes);
        } catch (DynamoConditionalCheckFailedException e) {
            log.warn(e.getMessage());
            return null;
        }
    }

    /**
     * UPDATE 処理を実行し、UPDATE前のレコードの内容を返します。
     * @return UPDATE前のレコード（今回更新しなかった値も含めて、全属性が入っている。レコードが存在しなかったら null）
     */
    public T executeAndGetOld() {
        UpdateItemResult updateItemResult = client.executeUpdate(this, ReturnValue.ALL_OLD);
        Map<String, AttributeValue> attributes = updateItemResult.getAttributes();
        if (log.isDebugEnabled() && attributes != null) {
            log.debug(attributes.toString());
        }
        return client.toRecord(table, attributes);
    }

    public UpdateItemRequest makeUpdateItemRequest(UpdateItemRequest request) {
        String updateExpression = makeUpdateExpression();
        request.withUpdateExpression(updateExpression);

//        request.withAttributeUpdates(table.createUpdateItem(values)); // TODO legacy parameter
        setupConditionalExpression(request);

        Map<String, String> expressionAttributeNames = makeAttibuteNames();
        request.withExpressionAttributeNames(expressionAttributeNames);
//        request.withExpressionAttributeValues(makeAttibuteValues());

        Map<String, AttributeValue> expressionAttributeValues = makeAttibuteValues();
        request.withExpressionAttributeValues(expressionAttributeValues);

        if (log.isDebugEnabled()) {
            String realTableName = client.getRealTableName(table);
            String conditionExpression = request.getConditionExpression();
            log.debug("new UpdateItemRequest().\nrealTableName = " + realTableName
                    + "\nupdateExpression = " + updateExpression + "\n"
                    + "expressionAttributeNames = " + expressionAttributeNames + "\n"
                    + "expressionAttributeValues = " + expressionAttributeValues + "\n"
                    + "conditionExpression = " + conditionExpression + "\n"
            );
        }


        // request.withUpdateExpression(updateExpression);
        return request;
    }


    private String makeUpdateExpression() {
        return table.createUpdateExpression(values);
    }

    private Map<String, String> makeAttibuteNames() {
        Map<String, String> map = new HashMap<>();
        int idx = 1;
        for (DynamoUpdateValue value : values) {
            map.put("#uname" + idx, value.getName());
            ++idx;
        }

        if (condition != null) {
            condition.appendAttributeNames(map, 1);
        }
        return map;
    }

    private Map<String, AttributeValue> makeAttibuteValues() {
        Map<String, AttributeValue> map = new HashMap<>();
        int idx = 1;
        for (DynamoUpdateValue value : values) {
            if (value.hasAttributeValue()) {
                map.put(":uval" + idx, value.getAttributeValue().getValue());
            }
            value.appendAdditionalAttributeValues(map);
            ++idx;
        }
        if (condition != null) {
            condition.appendAttributeValues(map, 1);
        }
        return map;
    }


    private void setupConditionalExpression(UpdateItemRequest request) {
        if (condition != null) {

            List<Integer> indexes = Arrays.asList(new Integer[]{1});
            String expression = condition.getExpression(indexes);
            request.withConditionExpression(expression);
/*
            Map<String, String> attrNames = new HashMap<>();
            condition.appendAttributeNames(attrNames, 1);

            Map<String, AttributeValue> attrValues = new HashMap<>();
            condition.appendAttributeValues(attrValues, 1);
*/
//            request.withExpressionAttributeNames(names);
        }
    }

    /**
     * 条件付き更新条件を設定します。
     * <b>この条件を設定したとき、条件を満たしていないと execute() 時に例外が発生します。</b>
     * 例外を発生させないようにするには、executeConditionalAndGet() を使ってください。
     * @param condition
     * @return
     */
    public DynamoUpdateChain condition(DynamoAttributeWithValue condition) {
        this.condition = condition;
        return this;
    }

}
