package jp.co.bizreach.jdynamo.admin;

import com.amazonaws.services.dynamodbv2.AmazonDynamoDBClient;
import com.amazonaws.services.dynamodbv2.model.CreateTableRequest;
import com.amazonaws.services.dynamodbv2.model.DeleteTableRequest;
import com.amazonaws.services.dynamodbv2.model.GetItemRequest;
import com.amazonaws.services.dynamodbv2.model.GetItemResult;
import com.amazonaws.services.dynamodbv2.model.KeySchemaElement;
import com.amazonaws.services.dynamodbv2.model.ListTablesResult;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughput;
import com.amazonaws.services.dynamodbv2.model.ResourceNotFoundException;
import jp.co.bizreach.jdynamo.DynamoClient;
import jp.co.bizreach.jdynamo.data.DynamoMetaTable;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.List;

/**
 * Created by iwami on 2016/07/05.
 */
@Slf4j
public class DynamoAdminClient {

    private AmazonDynamoDBClient amazonDynamoClient;

    private DynamoClient.DynamoClientPrivate clientPrivate;

    public DynamoAdminClient(AmazonDynamoDBClient amazonDynamoClient,
                             DynamoClient.DynamoClientPrivate clientPrivate) {
        this.amazonDynamoClient = amazonDynamoClient;
        this.clientPrivate = clientPrivate;
    }


    /**
     * テーブル名一覧を取得します。
     * TODO 100件以上のテーブルには未対応
     * @return
     */
    public List<String> getTableNames() {
        ListTablesResult listTablesResult = clientPrivate.getRawDynamoClient().listTables();
        List<String> tableNames = listTablesResult.getTableNames();
        if (log.isDebugEnabled()) {
            log.debug(StringUtils.join(tableNames));
        }
        return tableNames;
    }

    public List<String> getTableNamesAll() {
        ListTablesResult listTablesResult = clientPrivate.getRawDynamoClient().listTables();
        List<String> tableNames = listTablesResult.getTableNames();
        while (listTablesResult.getLastEvaluatedTableName() != null) {
            listTablesResult = clientPrivate.getRawDynamoClient().listTables(listTablesResult.getLastEvaluatedTableName());
            tableNames.addAll(listTablesResult.getTableNames());
        }
        return tableNames;
    }

    /**
     * Dynamo のテーブルを作成します。ローカル環境でのみ使って下さい。
     * @param table
     */
    public void createTableByLocal(DynamoMetaTable table) {
        amazonDynamoClient.createTable(makeCreateTableRequestByLocal(table));
    }

    public void deleteTableByLocal(DynamoMetaTable table) {
        try {
            amazonDynamoClient.deleteTable(makeDeleteTableRequestByLocal(table));
        } catch (ResourceNotFoundException e) {
            log.warn(e.getMessage(), e);
        }
    }

    private DeleteTableRequest makeDeleteTableRequestByLocal(DynamoMetaTable table) {
        return new DeleteTableRequest()
                .withTableName(clientPrivate.getRealTableName(table));
    }

    private CreateTableRequest makeCreateTableRequestByLocal(DynamoMetaTable table) {

        String realTableName = clientPrivate.getRealTableName(table);
        List<KeySchemaElement> keySchema = table.getKeySchema();
        List attributeDefinitions = table.getAttributeDefinitions();
        List globalSecondaryIndexes = table.createGlobalSecondaryIndexesByLocal();

        if (log.isDebugEnabled()) {
            log.debug("new CreateTableRequest().\nrealTableName = " + realTableName
                    + "\nattributeDefinitions" + attributeDefinitions
                    + "\nglobalSecondaryIndexes = " + globalSecondaryIndexes);

        }

        return new CreateTableRequest()
                .withTableName(realTableName)
                .withKeySchema(keySchema)
                .withGlobalSecondaryIndexes(globalSecondaryIndexes)
                .withAttributeDefinitions(attributeDefinitions)
                .withProvisionedThroughput(new ProvisionedThroughput(1L, 1L));
    }

    public GetItemResult getAsItemResult(DynamoMetaTable table, Object partitionKey, Object sortKey) {
        GetItemRequest request = new GetItemRequest()
                .withTableName(clientPrivate.getRealTableName(table))
                .withKey(clientPrivate.createKey(table, partitionKey, sortKey));
        GetItemResult itemResult = amazonDynamoClient.getItem(request);
        return itemResult;
    }


}
