package jp.co.bizreach.jdynamo.core;

import com.amazonaws.services.dynamodbv2.AmazonDynamoDBClient;
import com.amazonaws.services.dynamodbv2.model.BatchGetItemRequest;
import com.amazonaws.services.dynamodbv2.model.BatchGetItemResult;
import com.amazonaws.services.dynamodbv2.model.BatchWriteItemRequest;
import com.amazonaws.services.dynamodbv2.model.BatchWriteItemResult;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughputExceededException;
import com.amazonaws.services.dynamodbv2.model.PutItemRequest;
import com.amazonaws.services.dynamodbv2.model.QueryRequest;
import com.amazonaws.services.dynamodbv2.model.QueryResult;
import com.amazonaws.services.dynamodbv2.model.ScanRequest;
import com.amazonaws.services.dynamodbv2.model.ScanResult;
import jp.co.bizreach.jdynamo.DynamoRuntimeException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;

import java.util.Date;
import java.util.function.Supplier;

/**
 * スループット超過時にリトライ機構を実装するクラスです。
 * <p>
 * リトライは Exponential Backoff 形式で行われます。
 * sleep (ms) = random_between(100, 100 ** attempt)
 * </p>
 * <ul>
 *     <li>リトライは最大10回まで</li>
 *     <li>リトライの最大秒数を指定することも可</li>
 *     <li>秒数計算には JITTER（ランダム）を適用</li>
 * </ul>
 * Created by iwami on 2016/07/08.
 */
@Slf4j
public class DynamoThroughputAdjuster {

    private AmazonDynamoDBClient dynamoClient;

    private int maxRetries = 10; // デフォルトは10回までリトライ => 102300ms = 約102秒

    /** リトライ回数ではなくて秒数でリトライを制御したい場合、ここに値をセットしてください */
    private Long retryLimitMillisecond;

    private long retryStepMillisecond = 100L; // デフォルトは100, 200, 400ms の順でウェイト

    /** true にすると、リトライ間隔にランダム要素を入れる */
    private boolean jitter = true;

    /**
     * ライブラリが内部的に呼び出します。
     * <i>クライアントコードからは呼び出さないでください</i>
     * @param dynamoClient
     */
    public DynamoThroughputAdjuster(AmazonDynamoDBClient dynamoClient) {
        this.dynamoClient = dynamoClient;
    }

    public DynamoThroughputAdjuster(AmazonDynamoDBClient dynamoClient, long retryStepMillisecond, int maxRetries) {
        this.dynamoClient = dynamoClient;
        this.retryStepMillisecond = retryStepMillisecond;
        this.maxRetries = maxRetries;
    }

    /**
     * ライブラリが内部的に呼び出します。
     * <i>クライアントコードからは呼び出さないでください</i>
     * @param dynamoClient
     * @param retryLimitMillisecond
     */
    public DynamoThroughputAdjuster(AmazonDynamoDBClient dynamoClient, long retryLimitMillisecond) {
        this.dynamoClient = dynamoClient;
        this.retryLimitMillisecond = retryLimitMillisecond;
    }

    public QueryResult query(QueryRequest queryRequest) {
        return execWithRetry(() -> dynamoClient.query(queryRequest));
    }

    public ScanResult scan(ScanRequest scanRequest) {
        return execWithRetry(() -> dynamoClient.scan(scanRequest));
    }

    public void put(PutItemRequest putItemRequest) {
        execWithRetry(() -> dynamoClient.putItem(putItemRequest));
    }

    public BatchGetItemResult batchGet(BatchGetItemRequest request) {
        return execWithRetry(() -> dynamoClient.batchGetItem(request));
    }

    public BatchWriteItemResult batchPut(BatchWriteItemRequest request) {
        return execWithRetry(() -> dynamoClient.batchWriteItem(request));
    }

    private <T> T execWithRetry(Supplier<T> function) {
        int retries = 0;
        long startTime = new Date().getTime();
        while (true) {
            try {
                return function.get();
            } catch (ProvisionedThroughputExceededException e) {
                if (log.isDebugEnabled()) {
                    log.debug(e.getMessage(), e);
                }
            }
            checkRetryLimit(retries, startTime);
            sleepExponential(retries);
            ++retries;
        }
    }

    private void checkRetryLimit(int retries, long startTime) {
        if (retryLimitMillisecond != null) {
            // 時間でタイムアウト制限を付ける
            if (new Date().getTime() - startTime >= retryLimitMillisecond) {
                throw new DynamoRuntimeException(
                        "Retry has exceeded the limit " + retryLimitMillisecond + " millisecond.");
            }
        } else {
            // リトライ回数で制限を付ける
            if (retries >= maxRetries) {
                throw new DynamoRuntimeException("Retry count has exceeded the limit " + maxRetries + ".");
            }
        }
    }

    private void sleepExponential(int retries) {
        long waitTime = getWaitTimeExp(retries);
        if (log.isDebugEnabled()) {
            log.debug("Provisioned throughput has exceeded. Wait " + waitTime + " millisecond.");
        }
        try {
            Thread.sleep(waitTime);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
            Thread.currentThread().interrupt();
        }
    }

    private long getWaitTimeExp(int retryCount) {
        long waitTime = (long) Math.pow(2, retryCount) * retryStepMillisecond;
        return jitter ? RandomUtils.nextLong(retryStepMillisecond, waitTime + 1) : waitTime;
    }

}
