package jp.co.bizreach.jdynamo.data;

import com.amazonaws.services.dynamodbv2.model.AttributeDefinition;
import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.AttributeValueUpdate;
import jp.co.bizreach.jdynamo.DynamoRuntimeException;
import jp.co.bizreach.jdynamo.util.DynamoAttributeUtil;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.PropertyUtils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;

/**
 * Created by iwami on 2016/06/30.
 */
@Slf4j
public class DynamoAttributeDefinition {

    @Getter
    private String fieldName;

    @Getter
    @Setter
    private String dynamoAttrName;

    private DynamoAttributeType type;

    @Getter
    private DynamoMappingAttributeType mappingType;

    private Class<?> fieldType;

    public DynamoAttributeDefinition(Field field) {
        fieldName = field.getName();
        dynamoAttrName = field.getName();
        type = DynamoAttributeType.typeOf(field.getType());
        mappingType = DynamoMappingAttributeType.fieldOf(field);
        fieldType = field.getType();
    }

    public AttributeValue createAttributeValue(Object record) {
        AttributeValue attrValue = null;
        try {
            Object value = PropertyUtils.getProperty(record, fieldName);
            if (value == null) {
                return null;
            }
            attrValue = DynamoAttributeUtil.createAttributeValue(mappingType, value);
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            log.error(e.getMessage(), e);
        }

        return attrValue;
    }

    public Object getValue(Object record) {
        AttributeValueUpdate attrValue = null;
        try {
            Object value = PropertyUtils.getProperty(record, fieldName);
            return value;
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            throw new DynamoRuntimeException(e);
        }
    }

    public AttributeValueUpdate createAttributeValueUpdate(Object record) {
        AttributeValueUpdate attrValue = null;
        try {
            Object value = PropertyUtils.getProperty(record, fieldName);
            if (value == null) {
                return null;
            }
            attrValue = DynamoAttributeUtil.createAttributeValueUpdate(mappingType, value);
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            log.error(e.getMessage(), e);
        }


        return attrValue;
    }

    public void storeFieldByAttributeValue(Object record, String fieldName, AttributeValue value) {
        try {
            if (value == null) {
                PropertyUtils.setProperty(record, fieldName, null);
                return;
            }
            mappingType.storeField(record, fieldName, value);
        } catch (ReflectiveOperationException e) {
            log.error(e.getMessage(), e);
        } catch (Exception e) {
            // 想定していたのと違う型やデータだった場合
            log.error(e.getMessage(), e);
        }
    }

    public AttributeDefinition createAttributeDefinition(String dynamoName) {
        return mappingType.createAttributeDefinition(dynamoName);
    }
}
