package jp.co.bizreach.jdynamo.data;

import com.amazonaws.services.dynamodbv2.model.AttributeDefinition;
import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import com.amazonaws.services.dynamodbv2.model.AttributeValueUpdate;
import com.amazonaws.services.dynamodbv2.model.GlobalSecondaryIndex;
import com.amazonaws.services.dynamodbv2.model.KeySchemaElement;
import com.amazonaws.services.dynamodbv2.model.KeyType;
import com.amazonaws.services.dynamodbv2.model.Projection;
import com.amazonaws.services.dynamodbv2.model.ProjectionType;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughput;
import jp.co.bizreach.jdynamo.DynamoClientSetting;
import jp.co.bizreach.jdynamo.annotation.PartitionKey;
import jp.co.bizreach.jdynamo.annotation.SortKey;
import jp.co.bizreach.jdynamo.data.attr.DynamoAttributeSupport;
import jp.co.bizreach.jdynamo.data.attr.DynamoBooleanAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoCompressStringAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoDateAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoDateTimeAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoDecimalAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoIntegerAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoIntegerSetAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoLongAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoLongListAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoLongSetAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoStringAttribute;
import jp.co.bizreach.jdynamo.data.attr.DynamoStringSetAttribute;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.reflect.FieldUtils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Dynamo のメタテーブル情報を表す基底クラス
 * Created by iwami on 2016/06/30.
 */
@Slf4j
public class DynamoMetaTable<T> {

    @Getter
    private String baseTableName;

    @Getter
    private Class<?> recordClass;

    private List<KeySchemaElement> keySchema;
    private List<AttributeDefinition> attributeDefinitions;

    private String partitionKeyFieldName;
    private String sortKeyFieldName;

    @Getter
    private DynamoAttributeDefinition partitionKeyAttr;
    @Getter
    private DynamoAttributeDefinition sortKeyAttr;

    private List<DynamoAttributeDefinition> attributes;

    /** { dynamoAttrName -> definition } */
    private Map<String, DynamoAttributeDefinition> attributeDynamoMap;

    /** { fieldName -> definition } */
    private Map<String, DynamoAttributeDefinition> attributeFieldMap;

    private DynamoIndex mainIndex;

    private List<DynamoIndex> indexes;

    public DynamoMetaTable() {
        log.info("DynamoMetaTable::DynamoMetaTable");
        Type type = ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
        initialize((Class<?>) type);
    }

    public DynamoAttributeDefinition getAttributeDefinition(String fieldName) {
        return attributeFieldMap.get(fieldName);
    }

    private void initialize(Class<?> recordClass) {
        this.recordClass = recordClass;
        jp.co.bizreach.jdynamo.annotation.Table tableName = getClass().getAnnotation(jp.co.bizreach.jdynamo.annotation.Table.class);
        if (tableName != null) {
            baseTableName = tableName.baseName();
        }

        keySchema = new ArrayList<>();
        attributeDefinitions = new ArrayList<>();

        initByPartitionKey(recordClass);
        initBySortKey(recordClass);

        attributes = new ArrayList<>();
        attributeDynamoMap = new HashMap<>();
        attributeFieldMap = new HashMap<>();
        for (Field field : FieldUtils.getAllFields(recordClass)) {
            DynamoAttributeDefinition attr = new DynamoAttributeDefinition(field);
            if (attr.getMappingType() == DynamoMappingAttributeType.UNKNOWN) {
                log.info("Ignore field " + field + " because unknown mapping type.");
                continue;
            }
            attributes.add(attr);
            attributeFieldMap.put(field.getName(), attr);

            if (field.getName().equals(partitionKeyFieldName)) {
                partitionKeyAttr = attr;
            }
            if (field.getName().equals(sortKeyFieldName)) {
                sortKeyAttr = attr;
            }

        }

        indexes = new ArrayList<>();
        mainIndex = new DynamoIndex(this, null,
                getPartitionKeyField(recordClass), getSortKeyField(recordClass));
    }

    private Field getPartitionKeyField(Class<?> recordClass) {
        List<Field> partitionKeys = FieldUtils.getFieldsListWithAnnotation(recordClass, PartitionKey.class);
        if (CollectionUtils.isNotEmpty(partitionKeys)) {
            return partitionKeys.get(0);
        }
        return null;
    }

    private Field getSortKeyField(Class<?> recordClass) {
        List<Field> sortKeys = FieldUtils.getFieldsListWithAnnotation(recordClass, SortKey.class);
        if (CollectionUtils.isNotEmpty(sortKeys)) {
            return sortKeys.get(0);
        }
        return null;
    }

    private void initByPartitionKey(Class<?> recordClass) {
        List<Field> partitionKeys = FieldUtils.getFieldsListWithAnnotation(recordClass, PartitionKey.class);
        if (CollectionUtils.isNotEmpty(partitionKeys)) {
            Field field = partitionKeys.get(0);
            String name = field.getName();
            partitionKeyFieldName = name;
            keySchema.add(new KeySchemaElement(name, KeyType.HASH));

            addAttributeDefinitions(field, name);
        }
    }

    private void initBySortKey(Class<?> recordClass) {
        List<Field> sortKeys = FieldUtils.getFieldsListWithAnnotation(recordClass, SortKey.class);
        if (CollectionUtils.isNotEmpty(sortKeys)) {
            Field field = sortKeys.get(0);
            String name = field.getName();
            sortKeyFieldName = name;
            keySchema.add(new KeySchemaElement(name, KeyType.RANGE));

            addAttributeDefinitions(field, name);
        }
    }

    private AttributeDefinition addAttributeDefinitions(Field field, String dynamoName) {
        for (AttributeDefinition def : attributeDefinitions) {
            if (def.getAttributeName().equals(dynamoName)) {
                return null;
            }
        }

        DynamoAttributeDefinition attr = new DynamoAttributeDefinition(field);
        AttributeDefinition attributeDefinition = attr.createAttributeDefinition(dynamoName);
        if (attributeDefinition != null) {
            attributeDefinitions.add(attributeDefinition);
        }
        return attributeDefinition;
    }

    public List<KeySchemaElement> getKeySchema() {
        return keySchema;
    }

    public List<AttributeDefinition> getAttributeDefinitions() {
        return attributeDefinitions;
    }

    public Map<String,AttributeValue> createItem(Object record) {
        Map<String,AttributeValue> map = new HashMap<>();
        for (DynamoAttributeDefinition attr : attributes) {
            AttributeValue attributeValue = attr.createAttributeValue(record);
            if (attributeValue != null) {
                map.put(attr.getDynamoAttrName(), attributeValue);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug(map.toString());
        }
        return map;
    }

    public Map<String, AttributeValueUpdate> createUpdateItem(Object record) {
        Map<String,AttributeValueUpdate> map = new HashMap<>();
        for (DynamoAttributeDefinition attr : attributes) {
            AttributeValueUpdate attributeValue = attr.createAttributeValueUpdate(record);
            if (attributeValue != null) {
                map.put(attr.getDynamoAttrName(), attributeValue);
            }
        }
        if (log.isDebugEnabled()) {
            log.debug(map.toString());
        }
        return map;
    }

/*
    public Map<String, AttributeValueUpdate> createUpdateItem(List<DynamoUpdateValue> values) {
        Map<String, AttributeValueUpdate> map = new HashMap<>();
        for (DynamoUpdateValue value : values) {
            map.put(value.getName(), value.getAttributeValue());
        }
        if (log.isDebugEnabled()) {
            log.debug(map.toString());
        }
        return map;
    }
*/

    public DynamoIndex getMainIndex() {
        return mainIndex;
    }

    public void storeFieldByItem(Object record, Map.Entry<String, AttributeValue> entry, DynamoClientSetting clientSetting) {
        String dynamoAttrName = entry.getKey();

        AttributeValue value = entry.getValue();
        try {
            DynamoAttributeDefinition attribute = attributeDynamoMap.get(dynamoAttrName);

            if (attribute != null) {
                String fieldName = attribute.getFieldName();
                if (value == null) {
                    PropertyUtils.setProperty(record, fieldName, null);
                } else {
                    attribute.storeFieldByAttributeValue(record, fieldName, value);
                }
            } else {
                if (log.isWarnEnabled() && clientSetting.isNoWarnUnknownDynamoAttr() == false) {
                    log.warn(dynamoAttrName + " is not define.");
                }
            }
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            log.error(e.getMessage(), e);
        }
    }

    public void addIndex(DynamoIndex index) {
        indexes.add(index);

        String fieldName = index.getPartitionKeyFieldName();
        if (fieldName != null) {
            addAttributeDefinitions(FieldUtils.getDeclaredField(getRecordClass(), fieldName, true), fieldName);
        }

        fieldName = index.getSortKeyFieldName();
        if (fieldName != null) {
            addAttributeDefinitions(FieldUtils.getDeclaredField(getRecordClass(), fieldName, true), fieldName);
        }

    }

    public List<GlobalSecondaryIndex> createGlobalSecondaryIndexesByLocal() {
        List<GlobalSecondaryIndex> results = new ArrayList<>();
        for (DynamoIndex index : indexes) {
            if (index.getIndexName() == null) {
                continue;
            }

            GlobalSecondaryIndex tindex = new GlobalSecondaryIndex()
                    .withIndexName(index.getIndexName())
                    .withKeySchema(index.getKeySchema())
                    .withProjection(new Projection().withProjectionType(ProjectionType.ALL))
                    .withProvisionedThroughput(new ProvisionedThroughput(1L, 1L));
            results.add(tindex);
        }
        return CollectionUtils.isEmpty(results) ? null : results;
    }

    public DynamoKey createKey(T rec) {
        if (sortKeyAttr != null) {
            return new DynamoKey(partitionKeyAttr.getValue(rec), sortKeyAttr.getValue(rec));
        }
        return new DynamoKey(partitionKeyAttr.getValue(rec));
    }


    protected DynamoAttributeSupport createStringAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoStringAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createStringAttribute(String fieldName, String dynamoAttrName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        attribute.setDynamoAttrName(dynamoAttrName);
        DynamoAttributeSupport attr = new DynamoStringAttribute(attribute);
        attributeDynamoMap.put(dynamoAttrName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createCompressStringAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoCompressStringAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createLongAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoLongAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createLongAttribute(String fieldName, String dynamoAttrName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        attribute.setDynamoAttrName(dynamoAttrName);
        DynamoAttributeSupport attr = new DynamoLongAttribute(attribute);
        attributeDynamoMap.put(dynamoAttrName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createIntegerAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoIntegerAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createDecimalAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoDecimalAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createBooleanAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoBooleanAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createDateAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoDateAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createDateTimeAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoDateTimeAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }


    protected DynamoAttributeSupport createStringSetAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoStringSetAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createLongSetAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoLongSetAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createIntegerSetAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoIntegerSetAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    protected DynamoAttributeSupport createLongListAttribute(String fieldName) {
        DynamoAttributeDefinition attribute = attributeFieldMap.get(fieldName);
        DynamoAttributeSupport attr = new DynamoLongListAttribute(attribute);
        attributeDynamoMap.put(fieldName, attribute);
        return attr;
    }

    public String createUpdateExpression(List<DynamoUpdateValue> values) {
        StringBuilder sb = new StringBuilder();
//        int idx = 1;
        for (DynamoUpdateValue.UpdateCategory category : DynamoUpdateValue.UpdateCategory.values()) {
            List<DynamoUpdateValue> categoryValues = new ArrayList<>();
            for (DynamoUpdateValue updateValue : values) {
                DynamoUpdateValue.UpdateCategory updateCategory = updateValue.getUpdateCategory();
                if (category == updateCategory) {
                    categoryValues.add(updateValue);
                }
            }

            if (! categoryValues.isEmpty()) {
                if (sb.length() > 0) {
                    sb.append(" ");
                }
                sb.append(category.name()).append(" ");
                for (DynamoUpdateValue value : categoryValues) {
                    int idx = values.indexOf(value) + 1;
                    sb.append(value.createUpdateExpression(idx));
                    sb.append(", ");
//                    ++idx;
                }
                sb.setLength(sb.length() - 2);
            }
        }
        return sb.toString();
    }

}
