package jp.co.bizreach.jdynamo.data.attr;

import jp.co.bizreach.jdynamo.data.DynamoUpdateValue;

/**
 * 属性の Query, Filter, Uddate 条件を生成するときに使います。
 * Created by iwami on 2016/07/01.
 */
public interface DynamoAttributeSupport {

    /**
     * 属性値が targetValue と等しい条件を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue eq(Object targetValue);

    /**
     * 属性値が targetValue と等しくない条件を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue ne(Object targetValue);

    /**
     * 属性値（Set）が targetValue を含む条件を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue contains(Object targetValue);

    /**
     * 属性値が targetValues のいずれかと等しい条件を生成します。
     * @param targetValues
     * @return
     */
    DynamoAttributeWithValue in(Object... targetValues);

    /**
     * 属性値が targetValue よりも大きい条件（x ＞ value）を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue greaterThan(Object targetValue);

    /**
     * 属性値が targetValue よりも大きいか等しい条件（x ≧ value）を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue greaterThanEq(Object targetValue);

    /**
     * 属性値が targetValue よりも小さい条件（x ＜ value）を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue lessThan(Object targetValue);

    /**
     * 属性値が targetValue よりも小さいか等しい条件（x ≦ value）を生成します。
     * @param targetValue
     * @return
     */
    DynamoAttributeWithValue lessThanEq(Object targetValue);

    /**
     * 属性が存在する条件を生成します。
     * @return
     */
    DynamoAttributeWithValue exists();

    /**
     * 属性が存在しない条件を生成します。
     * @return
     */
    DynamoAttributeWithValue notExists();

    /**
     * 属性の値を指定値に更新します。
     * @param value 更新する値（null が指定された場合、その値は無視されます。属性を削除したい場合は clear を使います）
     * @return
     * @see DynamoAttributeSupport
     * @see #clear()
     */
    DynamoUpdateValue set(Object value);

    /**
     * 属性がまだ存在しない場合のみ、属性値を更新します。
     * 既に存在する場合は更新しません。
     * @param updateValue 属性が存在しない場合の属性更新値
     * @return
     */
    DynamoUpdateValue setIfNotExists(Object updateValue);

    /**
     * 属性の値をインクリメントします。
     * 属性が存在しない場合、AmazonDynamoDBException が発生します。
     * @param delta
     * @return
     */
    DynamoUpdateValue increment(Integer delta);

    /**
     * 属性の値を増加させます。
     * 属性が存在しない場合、値が delta で初期化されます。
     * @param delta
     * @return
     */
    DynamoUpdateValue add(Integer delta);

    /**
     * LIST属性の最後にNumber要素を追加します。
     * 属性が存在しない場合、サイズ1のLISTが作成されます。
     * @param value
     * @return
     */
    DynamoUpdateValue append(Long value);

    /**
     * SET属性の最後にString要素を追加します。
     * 属性が存在しない場合、サイズ1のSETが作成されます。
     * @param value
     * @return
     */
    DynamoUpdateValue append(String value);

    /**
     * SET属性から要素を削除します。
     * @param value
     * @return
     */
    DynamoUpdateValue delete(Long value);

    /**
     * 属性を削除します。
     * @return
     */
    DynamoUpdateValue clear();

    String getDynamoAttrName();

}
