package jp.co.bizreach.jdynamo.data.mapper;

import com.amazonaws.services.dynamodbv2.model.AttributeDefinition;
import com.amazonaws.services.dynamodbv2.model.AttributeValue;
import jp.co.bizreach.jdynamo.DynamoRuntimeException;
import jp.co.bizreach.jdynamo.annotation.Compress;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.beanutils.PropertyUtils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * Created by iwami on 2016/07/15.
 */
@Slf4j
public class DynamoCompressStringMapper implements IDynamoMapper {

    public static final String COMPRESS_CHARSET = "UTF-8";

    @Override
    public void appendValue(Map<String, AttributeValue> results, String key, Object value) {
        throw new DynamoRuntimeException("not support COMPRESS_STRING key.");
    }

    @Override
    public void storeField(Object record, String fieldName, AttributeValue value) throws ReflectiveOperationException {
        ByteBuffer bb = value.getB();
        if (bb == null) {
            // B に値が入っていなかったら、S に入っていないかチェック
            String svalue = value.getS();
            if (svalue != null) {
                PropertyUtils.setProperty(record, fieldName, svalue);
            }
            return;
        }
        try {
            PropertyUtils.setProperty(record, fieldName, createStringFromCompress(bb));
        } catch (IOException e) {
            throw new DynamoRuntimeException(e);
        }
    }

    @Override
    public boolean isMatch(Field field) {
        return String.class.equals(field.getType()) && field.getAnnotation(Compress.class) != null;
    }

    @Override
    public AttributeValue createAttributeValue(Object value) {
        AttributeValue attributeValue = new AttributeValue();
        try {
            attributeValue.withB(createCompressByteBuffer((String) value));
        } catch (IOException e) {
            throw new DynamoRuntimeException(e);
        }
        return attributeValue;
    }

    private String createStringFromCompress(ByteBuffer bb) throws IOException {
        byte[] bytes = bb.array();
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPInputStream is = new GZIPInputStream(bais);

        int chunkSize = 1024;
        byte[] buffer = new byte[chunkSize];
        int length;
        while ((length = is.read(buffer, 0, chunkSize)) != -1) {
            baos.write(buffer, 0, length);
        }

        String result = new String(baos.toByteArray(), COMPRESS_CHARSET);

        is.close();
        baos.close();
        bais.close();

        return result;
    }


    private static ByteBuffer createCompressByteBuffer(String value) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream os = new GZIPOutputStream(baos);
        os.write(value.getBytes(COMPRESS_CHARSET));
        os.close();
        baos.close();
        byte[] compressedBytes = baos.toByteArray();

        if (log.isDebugEnabled()) {
            log.debug("compressed length " + value.getBytes(COMPRESS_CHARSET).length + " -> " + compressedBytes.length);
        }

        ByteBuffer buffer = ByteBuffer.allocate(compressedBytes.length);
        buffer.put(compressedBytes, 0, compressedBytes.length);
        buffer.position(0);
        return buffer;
    }

    @Override
    public AttributeDefinition createAttributeDefinition(String dynamoName) {
        throw new DynamoRuntimeException("Compress String is not support.");
    }

    @Override
    public boolean isList() {
        return false;
    }

}
