package jp.co.bizreach.jdynamo.util;

import com.amazonaws.services.cloudwatch.AmazonCloudWatchClient;
import com.amazonaws.services.cloudwatch.model.Datapoint;
import com.amazonaws.services.cloudwatch.model.Dimension;
import com.amazonaws.services.cloudwatch.model.GetMetricStatisticsRequest;
import com.amazonaws.services.cloudwatch.model.GetMetricStatisticsResult;
import jp.co.bizreach.jdynamo.DynamoClient;
import jp.co.bizreach.jdynamo.data.DynamoMetaTable;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by iwami on 2016/07/08.
 */
@Slf4j
@AllArgsConstructor
public class DynamoCloudWatchClient {

    private DynamoClient.DynamoClientPrivate clientPrivate;

    private AmazonCloudWatchClient cloudWatchClient;

    /**
     * 直近24時間分の ConsumedReadCapacity を取得します。
     * @param table
     * @return 結果は時間の降順
     */
    public List<Datapoint> getRecentConsumedReadCapacityUnits(DynamoMetaTable table) {
        return getDynamoMetricDatapoints(table, "ConsumedReadCapacityUnits");
    }

    /**
     * 直近24時間分の ConsumedWriteCapacity を取得します。
     * @param table
     * @return 結果は時間の降順
     */
    public List<Datapoint> getRecentConsumedWriteCapacityUnits(DynamoMetaTable table) {
        return getDynamoMetricDatapoints(table, "ConsumedWriteCapacityUnits");
    }

    private List<Datapoint> getDynamoMetricDatapoints(DynamoMetaTable table, String metricName) {
        AmazonCloudWatchClient client = cloudWatchClient;
        GetMetricStatisticsRequest statisticsRequest = new GetMetricStatisticsRequest();
        statisticsRequest.setNamespace("AWS/DynamoDB");
        statisticsRequest.setMetricName(metricName);
        statisticsRequest.setStartTime(new Date(new Date().getTime() - 1000 * 60 * 60 * 24));
        statisticsRequest.setEndTime(new Date());
        statisticsRequest.setPeriod(60);
        statisticsRequest
                .setStatistics(Arrays.asList("Sum", "Maximum", "Minimum", "SampleCount", "Average"));

        Collection<Dimension> dimensions = new ArrayList<>();
        Dimension dimension = new Dimension().withName("Table").withValue(clientPrivate.getRealTableName(table));
        dimensions.add(dimension);

        statisticsRequest.setDimensions(dimensions);

        GetMetricStatisticsResult statisticsResult = client.getMetricStatistics(statisticsRequest);
        List<Datapoint> datapoints = statisticsResult.getDatapoints();
        Stream<Datapoint> sorted = datapoints.stream().sorted(Comparator.comparing(Datapoint::getTimestamp).reversed());
        return sorted.collect(Collectors.toList());
    }

}
