package kz.greetgo.script.model.context.act;

import java.util.ArrayList;
import java.util.List;
import kz.greetgo.script.model.expr.flow.ExprAct;
import kz.greetgo.script.model.translate.ValueExtType;
import kz.greetgo.script.model.update.ActArgDisplayName;
import kz.greetgo.script.model.update.ActDisplayNames;
import lombok.Builder;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;

/**
 * Содержит всю определяющую информацию о динамическом акте.
 * <p>
 * Динамический акт обслуживает специальный интерфейс
 */
@Builder
@RequiredArgsConstructor
public class ActDynamicDefinition implements ActDefinition {

  /**
   * Отображаемое имя, чтобы человек мог её идентифицировать
   */
  public final @NonNull String displayName;

  /**
   * Полное Фамилия Имя Отчество автора, кто определил данный акт
   */
  public final @NonNull String author;

  /**
   * Описание акта
   */
  public final @NonNull String description;

  /**
   * Тип исходного выражения, к которому можно применить данный акт
   * <p>
   * Его идентификатор в скрипте находиться здесь: {@link ExprAct#leftExprId}.
   */
  public final @NonNull ValueExtType srcType;

  /**
   * Тип результата применения данного акта.
   * <p>
   * Может быть null - тогда акт не требует значения, т.е. читать его нельзя, а при записи он не требует значения. Другими словами это метод void
   */
  public final ValueExtType resultValueType;

  /**
   * Описание результата акта
   */
  public final String resultValueDescription;

  /**
   * Аргументы, передаваемые в метод
   */
  public final List<ActArgDynDef> arguments;

  /**
   * Отменить запись по этому акту.
   * <p>
   * true - записывать в акт нельзя, иначе - можно
   */
  public final boolean cancelWrite;

  /**
   * Можно ли использовать чтение по данному акту
   * <p>
   * true - читать акт нельзя, иначе - можно
   */
  public final boolean cancelRead;

  @Override
  public String toString() {
    return "ActDynamicDefinition{" +
      "srcType=" + srcType +
      ", displayName='" + displayName + '\'' +
      ", resultValueType=" + resultValueType +
      '}';
  }

  @Override
  public boolean cancelWrite() {
    return cancelWrite;
  }

  @Override
  public boolean cancelRead() {
    return cancelRead;
  }

  @Override
  public ValueExtType resultValueType() {
    return resultValueType;
  }

  @Override
  public ActDisplayNames toDisplayNames() {
    ActDisplayNames ret = new ActDisplayNames();
    ret.displayName = displayName;

    if (arguments != null) {
      ret.args = new ArrayList<>();
      for (final ActArgDynDef argument : arguments) {
        final ActArgDisplayName a = new ActArgDisplayName();
        a.argId      = argument.argId;
        a.valueType  = argument.valueType;
        a.nameBefore = argument.nameBefore;
        a.nameAfter  = argument.nameAfter;
        ret.args.add(a);
      }
    }

    return ret;
  }
}
