package kz.greetgo.script.model.expr.flow;

import java.math.BigDecimal;
import kz.greetgo.kafka.model.KafkaModel;
import kz.greetgo.mybpm_util_light.ann.dumping.MixingId;
import kz.greetgo.script.model.context.model.BoiRef;
import kz.greetgo.script.model.context.model.BoiRefCode;
import kz.greetgo.script.model.expr.Expr;
import kz.greetgo.script.model.expr.flow.value.ExprValueType;
import kz.greetgo.script.model.expr.flow.value.GeoPointShowType;
import kz.greetgo.script.model.translate.ValueExtType;
import lombok.ToString;
import lombok.experimental.FieldNameConstants;

@ToString
@KafkaModel
@FieldNameConstants
public class ExprValue extends Expr {
  /**
   * Определяет тип данного выражения
   */
  public ExprValueType exprValueType;

  /**
   * Определяет тип выражения константы. Используется, если exprValueType == CONST
   */
  public ValueExtType valueType;

  /**
   * Определяет идентификатор блока, где объявлена переменная. Используется, если exprValueType == VAR_REF
   */
  public String varBlockId;

  /**
   * Определяет тип константы.
   * <p>
   * Это поле определяет основное поведение компонента тем, что определяет основную стратегию на сервере.
   */
  public String constType;

  /**
   * Значение выбранное для типа: constType
   * <hr>
   * Если поле constType == boolean,
   * то данное поле принимает значения: `yes` или `no`.
   * Если в нём оказывается пусто или любое другое значение, система воспринимает его как `no`.
   * <hr>
   * Если поле constType == Date,
   * то данное поле принимает значения даты и времени в ISO формате для UTC (например: `2020-01-23T10:11:21.003Z`).
   * Так же система воспринимает пустое значение как отсутствие даты и времени.
   * Любое другое значение система воспринимает как пустое.
   * <hr>
   * Если поле constType == BigDecimal,
   * то данное поле принимает значения числа в стандартной форме: <code>[+-]?d+([.,]d+([eE]d+)?)?</code>.
   * Могут произвольно встречаться пробелы и знаки подчёркивания (_) - они просто игнорируются.
   * Любое другое значение система воспринимает как 0.
   * <hr>
   * Если поле constType == String, то данное поле хранит текст
   */
  public String value;

  /**
   * Код бизнес-объекта
   */
  public String boCode;

  /**
   * Код поля бизнес-объекта
   */
  public String fieldCode;

  /**
   * Ссылка на инстанцию бизнес-объекта
   */
  @MixingId
  public String boiId;

  /**
   * Дескриптор фабрики по созданию Java-объектов
   * <p>
   * Это поле используется, если constType.baseType == JavaObjectFactories
   */
  public String objectDescriptor;

  /**
   * Указывает ввода текста в несколько строк.
   * <p>
   * Если true, то текст вводиться в несколько строк.
   * <p>
   * Если false, то тест вводиться в одну строку.
   * <p>
   * Используется, если constType.baseType == String
   */
  public boolean isTextMultiline;

  /**
   * Ширина многострочного поля ввода в виде стиля (например: 540px)
   */
  public String textareaWidth;

  /**
   * Высота многострочного поля ввода в виде стиля (например: 310px)
   */
  public String textareaHeight;

  /**
   * Широта в градусах от -90 (юг), через 0 - экватор, до +90 (север).
   * <p>
   * Линия с одинаковой широтой - параллель. Самая большая параллель - экватор.
   */
  public double latitude;

  /**
   * Долгота в градусах от -180 (с запада), через 0 - гринвич, до +180 (на восток)
   * <p>
   * Линия с одинаковой долготой - меридиан
   */
  public double longitude;

  /**
   * Тип отображения географических координат
   */
  public GeoPointShowType geoPointShowType;

  /**
   * Идентификатор опции выбора для SingleSelectRef
   */
  public              String optionId;
  public static final String JavaObjectFactories = "JavaObjectFactories";

  /**
   * Формирует выражения для обращения к этому процессу
   * <p>
   * В скрипте возвращает {@link BoiRef}
   *
   * @return выражение для обращения к этому процессу
   */
  public static ExprValue thisProcess() {
    ExprValue ret = new ExprValue();
    ret.exprValueType = ExprValueType.THIS_PROCESS;
    return ret;
  }

  public static ExprValue text(String text) {
    ExprValue ret = new ExprValue();
    ret.exprValueType = ExprValueType.CONST;
    ret.valueType     = ValueExtType.text();
    ret.constType     = String.class.getSimpleName();
    ret.value         = text;
    return ret;
  }


  public static ExprValue varRef(String varBlockId) {
    ExprValue ret = new ExprValue();
    ret.exprValueType = ExprValueType.VAR_REF;
    ret.varBlockId    = varBlockId;
    return ret;
  }

  public static ExprValue boiRefCode(BoiRefCode boiRefCode) {
    ExprValue ret = new ExprValue();
    ret.exprValueType = ExprValueType.CONST;
    ret.valueType     = ValueExtType.boiRefCode(boiRefCode.boCode);
    ret.constType     = BoiRefCode.class.getSimpleName();
    ret.boCode        = boiRefCode.boCode;
    ret.boiId         = boiRefCode.boiId;
    return ret;
  }

  public static Expr number(String number) {
    ExprValue ret = new ExprValue();
    ret.exprValueType = ExprValueType.CONST;
    ret.valueType     = ValueExtType.number();
    ret.constType     = BigDecimal.class.getSimpleName();
    ret.value         = number;
    return ret;
  }

  public static ExprValue javaObjectFactory(String objectDescriptor, ValueExtType resultType) {
    ExprValue ret = new ExprValue();
    ret.exprValueType    = ExprValueType.CONST;
    ret.constType        = ExprValue.JavaObjectFactories;
    ret.valueType        = resultType;
    ret.objectDescriptor = objectDescriptor;
    return ret;
  }
}
