package kz.greetgo.script.model.expr.flow.act;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.RequiredArgsConstructor;
import lombok.ToString;

import static java.util.Objects.requireNonNull;
import static kz.greetgo.script.model.expr.flow.act.ActIdUtil.checkOnCorrectChars;

@ToString
@EqualsAndHashCode
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public class ActId {
  public static final String KIND          = "K";// K - Kind
  public static final String BASE_TYPE     = "T";// T - Type
  public static final String FIELD_CODE    = "F";// F - Field
  public static final String PLUGIN_ID     = "P";// P - Plugin
  public static final String ACT_SOURCE_ID = "S";// S - Source
  ///////////////////////////////////////////////// Z - Земля
  ///////////////////////////////////////////////// V - Victory
  ///////////////////////////////////////////////// O - Orion

  private static final Set<String> REGISTERED_PARAM_NAMES = Set.of(KIND, BASE_TYPE, FIELD_CODE, PLUGIN_ID, ACT_SOURCE_ID);

  public final Map<String, String> data;

  public static ActId fix(String baseType, String fieldCode) {
    requireNonNull(baseType, "g3IPgGwY63 :: baseType");
    requireNonNull(fieldCode, "QoHvuKoWNQ :: fieldCode");
    checkOnCorrectChars(baseType, "m25q79x5Fw :: baseType");
    checkOnCorrectChars(fieldCode, "pxAZ3W6cS7 :: fieldCode");

    return new ActId(Map.of(
      KIND, ActIdKind.FIX.name(),
      BASE_TYPE, baseType,
      FIELD_CODE, fieldCode
    ));
  }

  public static ActId dynamic(String sourcePluginId, String actSourceId, String fieldCode, Map<String, String> params) {
    requireNonNull(fieldCode, "ZxUhp3NE74 :: fieldCode");
    requireNonNull(actSourceId, "iS26mj1k0t :: actSourceId");
    checkOnCorrectChars(fieldCode, "yenZQqL1XG :: fieldCode");
    checkOnCorrectChars(actSourceId, "EaEM3oBxZz :: actSourceId");
    checkParamNames("82RG1oy3Cr", params);
    if (sourcePluginId != null && sourcePluginId.length() == 0) {
      sourcePluginId = null;
    }
    checkOnCorrectChars(sourcePluginId, "x0omjVYd7E :: sourcePluginId");

    Map<String, String> map = new HashMap<>(params);
    map.putAll(ActLocalId.dyn(fieldCode).data);
    map.putAll(ActSourceId.of(sourcePluginId, actSourceId).data);
    map.put(KIND, ActIdKind.DYN.name());

    return new ActId(Map.copyOf(map));
  }

  public static ActId dynamic(ActSourceId actSourceId, ActLocalId localId) {
    Map<String, String> data     = new HashMap<>(localId.data);
    String              pluginId = actSourceId.pluginId();
    if (pluginId != null) {
      data.put(PLUGIN_ID, pluginId);
    }
    String localSourceId = actSourceId.localSourceId();
    data.put(ACT_SOURCE_ID, localSourceId);
    data.put(KIND, ActIdKind.DYN.name());
    return new ActId(data);
  }

  @SuppressWarnings("SameParameterValue")
  static void checkParamNames(String placeId, Map<String, String> params) {
    for (final String paramName : params.keySet()) {
      checkOnCorrectChars(paramName, placeId + " :: " + paramName);
      for (final String registeredParamName : REGISTERED_PARAM_NAMES) {
        if (registeredParamName.equals(paramName)) {
          throw new RuntimeException(placeId + " :: You cannot use param name `" + registeredParamName + "`"
                                       + " because it was registered for system use. Please use another param name."
                                       + " System parameters are: " + String.join(", ", REGISTERED_PARAM_NAMES));
        }
      }
    }
  }

  public static ActId dynamic(String sourcePluginId, String actSourceId, String fieldCode) {
    return dynamic(sourcePluginId, actSourceId, fieldCode, Map.of());
  }

  public String strValue() {
    return ActIdUtil.strValue(data);
  }

  private static Optional<ActId> checkAndCreate(Map<String, String> data, String... names) {
    for (final String name : names) {
      if (!data.containsKey(name)) {
        return Optional.empty();
      }
    }
    return Optional.of(new ActId(Map.copyOf(data)));
  }

  public static Optional<ActId> parse(String strValue) {
    if (strValue == null) {
      return Optional.empty();
    }

    Map<String, String> data = ActIdUtil.parse(strValue);

    ActIdKind kind = ActIdKind.parse(data.get(KIND)).orElse(null);
    if (kind == null) {
      return Optional.empty();
    }

    switch (kind) {

      case FIX:
        return checkAndCreate(data, BASE_TYPE, FIELD_CODE);

      case DYN:
        return checkAndCreate(data, FIELD_CODE, ACT_SOURCE_ID);

      default:
        return Optional.empty();
    }
  }

  public ActIdKind kind() {
    return ActIdKind.valueOf(data.get(KIND));
  }

  public String baseType() {
    return data.get(BASE_TYPE);
  }

  public String fieldCode() {
    return data.get(FIELD_CODE);
  }

  public String pluginId() {
    return data.get(PLUGIN_ID);
  }

  public String localActSourceId() {
    return data.get(ACT_SOURCE_ID);
  }

  public String get(String paramName) {
    return data.get(paramName);
  }

  public ActSourceId actSourceId() {
    return ActSourceId.fromData(data);
  }

  public ActLocalId localId() {
    return ActLocalId.from(this);
  }

  public Map<String, String> restParams() {
    return data.entrySet()
               .stream()
               .filter(e -> !(REGISTERED_PARAM_NAMES.contains(e.getKey())))
               .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
  }
}
