package kz.greetgo.script.model.translate;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;
import java.util.Set;
import kz.greetgo.kafka.model.KafkaModel;
import kz.greetgo.script.ann.model.ValueType;
import kz.greetgo.script.model.context.model.BoRef;
import kz.greetgo.script.model.context.model.BoRefCode;
import kz.greetgo.script.model.context.model.BoiFieldRef;
import kz.greetgo.script.model.context.model.BoiFieldRefCode;
import kz.greetgo.script.model.context.model.BoiRef;
import kz.greetgo.script.model.context.model.BoiRefCode;
import kz.greetgo.script.model.context.model.GeoPoint;
import kz.greetgo.script.model.context.model.Period;
import kz.greetgo.script.model.context.model.SingleSelectRef;
import kz.greetgo.script.model.context.model.SingleSelectRefCode;
import kz.greetgo.script.model.context.model.signature.MybpmFile;
import lombok.EqualsAndHashCode;
import lombok.experimental.FieldNameConstants;

@KafkaModel
@EqualsAndHashCode
@FieldNameConstants
public class ValueExtType {

  /**
   * Тип основа типа выражения
   * <p>
   * Но также может формироваться другими способами
   */
  public String baseType;

  /**
   * Это поле определяет лишь тип бордюра (в новой концепции)
   */
  public ValueType type;

  /**
   * Признак массива (ну что данный тип является массивом)
   */
  public boolean isArray;

  /**
   * Идентификатор бизнес-объекта
   */
  public String boId;

  /**
   * Код бизнес-объекта
   */
  public String boCode;

  /**
   * Идентификатор поля бизнес-объекта
   */
  public String fieldId;

  /**
   * Код поля бизнес-объекта
   */
  public String fieldCode;

  public static ValueExtType of(String baseType, ValueType type) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = baseType;
    ret.type     = type;
    return ret;
  }

  public static ValueExtType ofClass(Class<?> baseTypeClass) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = baseTypeClass.getSimpleName();
    ret.type     = ValueType.Object;
    return ret;
  }

  public static ValueExtType arrayOfClass(Class<?> baseTypeClass) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = baseTypeClass.getSimpleName();
    ret.type     = ValueType.Object;
    ret.isArray  = true;
    return ret;
  }

  public static ValueExtType number() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = BigDecimal.class.getSimpleName();
    ret.type     = ValueType.Number;
    return ret;
  }

  public static ValueExtType mybpmFile() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = MybpmFile.class.getSimpleName();
    ret.type     = ValueType.File;
    ret.isArray  = true;
    return ret;
  }

  public static ValueExtType text() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = String.class.getSimpleName();
    ret.type     = ValueType.Text;
    return ret;
  }

  public static ValueExtType date() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = Date.class.getSimpleName();
    ret.type     = ValueType.Date;
    return ret;
  }

  public static ValueExtType geoPoint() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = GeoPoint.class.getSimpleName();
    ret.type     = ValueType.GeoPoint;
    return ret;
  }

  public static ValueExtType bool() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = Boolean.class.getSimpleName();
    ret.type     = ValueType.Bool;
    return ret;
  }

  public static ValueExtType period() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = Period.class.getSimpleName();
    ret.type     = ValueType.Period;
    return ret;
  }

  public static ValueExtType boiRef() {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = BoiRef.class.getSimpleName();
    ret.type     = ValueType.Bo;
    return ret;
  }

  public static ValueExtType boiRefCode() {
    return boiRefCode(null);
  }

  public static ValueExtType boiRefCode(String boCode) {
    return boiRefCode(boCode, false);
  }

  public static ValueExtType boiRefCode(String boCode, boolean isArray) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = BoiRefCode.class.getSimpleName();
    ret.type     = ValueType.Bo;
    ret.boCode   = boCode;
    ret.isArray  = isArray;
    return ret;
  }

  public static ValueExtType boiFieldRefCode() {
    return boiFieldRefCode(null, null);
  }


  public static ValueExtType boiFieldRef(String boId, String fieldId) {
    ValueExtType ret = new ValueExtType();
    ret.baseType = BoiFieldRef.class.getSimpleName();
    ret.type     = ValueType.BoField;
    ret.boId     = boId;
    ret.fieldId  = fieldId;
    return ret;
  }

  public static ValueExtType boiFieldRefCode(String boCode, String fieldCode) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType  = BoiFieldRefCode.class.getSimpleName();
    ret.type      = ValueType.Co;
    ret.boCode    = boCode;
    ret.fieldCode = fieldCode;
    return ret;
  }

  @SuppressWarnings("unused")
  public static ValueExtType boiFieldRefCode(String boCode, String fieldCode, boolean isArray) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType  = BoiFieldRefCode.class.getSimpleName();
    ret.type      = ValueType.Bo;
    ret.boCode    = boCode;
    ret.fieldCode = fieldCode;
    ret.isArray   = isArray;
    return ret;
  }

  public static ValueExtType boRef(String boId) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = BoRef.class.getSimpleName();
    ret.boId     = boId;
    ret.type     = ValueType.Bo;
    return ret;
  }

  public static ValueExtType boRefCode(String boCode) {
    final ValueExtType ret = new ValueExtType();
    ret.baseType = BoRefCode.class.getSimpleName();
    ret.boCode   = boCode;
    ret.type     = ValueType.Bo;
    return ret;
  }

  public static ValueExtType singleSelect(String boId, String fieldId) {
    ValueExtType ret = new ValueExtType();
    ret.type     = ValueType.SingleSelect;
    ret.baseType = SingleSelectRef.class.getSimpleName();
    ret.boId     = boId;
    ret.fieldId  = fieldId;
    ret.isArray  = false;
    return ret;
  }

  public static ValueExtType singleSelectCode(String boCode, String fieldCode) {
    ValueExtType ret = new ValueExtType();
    ret.type      = ValueType.SingleSelect;
    ret.baseType  = SingleSelectRefCode.class.getSimpleName();
    ret.boCode    = boCode;
    ret.fieldCode = fieldCode;
    ret.isArray   = false;
    return ret;
  }

  public ValueExtType asAlone() {
    ValueExtType ret = new ValueExtType();
    ret.baseType  = baseType;
    ret.type      = type;
    ret.isArray   = false;
    ret.boId      = boId;
    ret.coId      = coId;
    ret.boCode    = boCode;
    ret.fieldCode = fieldCode;
    ret.fieldId   = fieldId;
    return ret;
  }

  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder(getClass().getSimpleName() + '{');
    sb.append(isArray ? "ARR " : "ONE ");
    if (baseType != null) {
      sb.append(baseType);
    } else if (type != null) {
      sb.append("<").append(type).append(">");
    }
    if (boId != null) {
      sb.append(", boId=").append(boId);
    }
    if (coId != null) {
      sb.append(", coId=").append(coId);
    }
    if (boCode != null) {
      sb.append(", boCode=").append(boCode);
    }
    if (fieldCode != null) {
      sb.append(", fieldCode=").append(fieldCode);
    }
    if (fieldId != null) {
      sb.append(", fieldId=").append(fieldId);
    }
    if (containingBoIds != null) {
      sb.append(", containingBoIds=").append(containingBoIds);
    }
    return sb.append('}').toString();
  }

  //
  //
  // Поля ниже - устаревшие и подлежащие удалению

  public String      coId;
  public Set<String> containingBoIds;
  public String      instanceBoId;
  public String      instanceBoiId;
  public String      instanceBoiName;
  public String      objectId;

  public String enumNativeName;

  public boolean isPersonBo;

  public boolean isNative;

  public static ValueExtType single(ValueType valueType) {
    final ValueExtType ret = new ValueExtType();
    ret.type     = valueType;
    ret.isArray  = false;
    ret.baseType = tryDetectBaseTypeByValueType(valueType);
    return ret;
  }

  private static String tryDetectBaseTypeByValueType(ValueType valueType) {
    if (valueType == null) {
      return null;
    }
    switch (valueType) {

      case Bo:
      case Co:
        return BoiRef.class.getSimpleName();

      case Bool:
        return Boolean.class.getSimpleName();

      case Date:
        return Date.class.getSimpleName();

      case File:
        return ValueExtType.mybpmFile().baseType;

      case Text:
        return String.class.getSimpleName();

      case Number:
        return BigDecimal.class.getSimpleName();

      case Period:
        return Period.class.getSimpleName();

      case BoField:
        return BoiFieldRef.class.getSimpleName();

      case GeoPoint:
        return GeoPoint.class.getSimpleName();

      case SingleSelect:
        return SingleSelectRef.class.getSimpleName();

      default:
        return null;
    }
  }


  public static ValueExtType multiple(ValueType valueType) {
    final ValueExtType ret = new ValueExtType();
    ret.type     = valueType;
    ret.isArray  = true;
    ret.baseType = tryDetectBaseTypeByValueType(valueType);
    return ret;
  }

  public static ValueExtType bo(String boId, boolean isArray) {
    final ValueExtType ret = new ValueExtType();
    ret.type    = ValueType.Bo;
    ret.isArray = isArray;
    ret.boId    = boId;
    return ret;
  }

  public static ValueExtType co(String coId, boolean isArray, Set<String> containingBoIds) {
    final ValueExtType ret = new ValueExtType();
    ret.type            = ValueType.Co;
    ret.isArray         = isArray;
    ret.coId            = coId;
    ret.containingBoIds = containingBoIds;
    return ret;
  }

  public static ValueExtType enumRef(String enumNativeName) {
    ValueExtType single = single(ValueType.EnumRef);
    single.enumNativeName = enumNativeName;
    single.baseType       = enumNativeName;
    return single;
  }

  public static ValueExtType tab() {
    return single(ValueType.Tab);
  }

  public static ValueExtType object(String objectId) {
    ValueExtType single = single(ValueType.Object);
    single.objectId = objectId;
    single.baseType = objectId;
    return single;
  }

  public static boolean eq(ValueExtType extType1, ValueExtType extType2) {
    if (extType1 == null && extType2 == null) {
      return true;
    }
    if (extType1 == null || extType2 == null) {
      return false;
    }

    ValueType type = extType1.type;

    if (type != extType2.type) {
      if (type == ValueType.Bo && extType2.type == ValueType.Co) {

        return extType2.containingBoIds.contains(extType1.boId);

      } else if (type == ValueType.Co && extType2.type == ValueType.Bo) {

        return extType1.containingBoIds.contains(extType2.boId);

      } else {
        return false;
      }
    }

    if (type == null) {
      return true;
    }

    switch (type) {

      case Bo:
        return Objects.equals(extType1.boId, extType2.boId);

      case Co:
        return Objects.equals(extType1.boId, extType2.boId) && Objects.equals(extType1.containingBoIds, extType2.containingBoIds);

      case Object:
        return Objects.equals(extType1.objectId, extType2.objectId);

      case EnumRef:
        return Objects.equals(extType1.enumNativeName, extType2.enumNativeName);

      case BoField:
        return Objects.equals(extType1.boId, extType2.boId) && Objects.equals(extType1.fieldId, extType2.fieldId);

      case SingleSelect:
        return true;

      default:
        return true;
    }
  }

  @SuppressWarnings("BooleanMethodIsAlwaysInverted")
  public static boolean eqBoCo(ValueExtType extType1, ValueExtType extType2) {

    if (extType1 == null && extType2 == null) {
      return true;
    }
    if (extType1 == null || extType2 == null) {
      return false;
    }

    ValueType type = extType1.type;

    if (type != extType2.type) {
      if (type == ValueType.Bo && extType2.type == ValueType.Co) {

        return Objects.equals(extType1.boId, extType2.coId) || extType2.containingBoIds.contains(extType1.boId);

      } else if (type == ValueType.Co && extType2.type == ValueType.Bo) {

        return Objects.equals(extType1.coId, extType2.boId) || extType1.containingBoIds.contains(extType2.boId);

      }
    }

    return eq(extType1, extType2);
  }
}
