package kz.greetgo.script.model.definitions.enums.extractor;

import java.lang.annotation.Annotation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import kz.greetgo.script.ann.enums.EnumDescription;
import kz.greetgo.script.ann.enums.EnumName;
import kz.greetgo.script.ann.model.ValueType;
import kz.greetgo.script.model.context.ScriptObjectContext;
import kz.greetgo.script.model.definitions.enums.EnumDefinition;
import kz.greetgo.script.model.definitions.enums.EnumValueDefinition;
import kz.greetgo.script.model.definitions.enums.EnumsDefinition;
import kz.greetgo.script.model.definitions.func.FuncDefinition;
import kz.greetgo.script.model.definitions.func.FuncGroupsDefinition;
import kz.greetgo.script.model.definitions.func.extractor.FuncGroupsDefinitionExtractor;
import kz.greetgo.script.model.definitions.object.ObjectMemberDefinition;
import kz.greetgo.script.model.definitions.object.ObjectsDefinition;
import kz.greetgo.script.model.definitions.object.extractor.ObjectsDefinitionExtractor;
import kz.greetgo.script.model.translate.ValueExtType;
import lombok.SneakyThrows;

public class EnumsDefinitionExtractor {

  private static final Map<Class<? extends ScriptObjectContext>, EnumsDefinition> cached = new ConcurrentHashMap<>();

  public static void init(Class<? extends ScriptObjectContext> scriptObjectContextClass) {
    loadEnumsDefinition(scriptObjectContextClass);
  }

  public static EnumsDefinition loadEnumsDefinition(Class<? extends ScriptObjectContext> scriptObjectContextClass) {

    {
      EnumsDefinition fromCache = cached.get(scriptObjectContextClass);
      if (fromCache != null) {
        return fromCache;
      }
    }

    synchronized (cached) {
      EnumsDefinition fromCache = cached.get(scriptObjectContextClass);
      if (fromCache != null) {
        return fromCache;
      }

      EnumsDefinition load = load(scriptObjectContextClass);

      cached.put(scriptObjectContextClass, load);

      return load;
    }

  }

  @SuppressWarnings("unused")
  private static <T extends ScriptObjectContext> EnumsDefinition load(Class<? extends ScriptObjectContext> scriptObjectContextClass) {

    EnumsDefinition enumsDefinition = new EnumsDefinition();

    ObjectsDefinition    objectsDefinition    = ObjectsDefinitionExtractor.loadObjectsDefinition(scriptObjectContextClass);
    FuncGroupsDefinition funcGroupsDefinition = FuncGroupsDefinitionExtractor.loadFuncGroupsDefinition(scriptObjectContextClass);

    funcGroupsDefinition.groups.values().stream()
                               .flatMap(x -> x.functions.values().stream())
                               .flatMap(EnumsDefinitionExtractor::toEnumDefinitions)
                               .forEach(x -> enumsDefinition.values.put(x.nativeName, x));

    objectsDefinition.objects.values().stream()
                             .flatMap(x -> x.members.values().stream())
                             .flatMap(EnumsDefinitionExtractor::toEnumDefinitions)
                             .forEach(x -> enumsDefinition.values.put(x.nativeName, x));

    return enumsDefinition;

  }

  @SneakyThrows
  private static Stream<EnumDefinition> toEnumDefinitions(FuncDefinition funcDefinition) {

    List<EnumDefinition> list = new ArrayList<>();

    if (funcDefinition.returnDefinition != null && funcDefinition.returnDefinition.type.type == ValueType.EnumRef) {

      list.add(
        generateEnumDefinition(funcDefinition.returnDefinition.enumFullClassName,
                               funcDefinition.returnDefinition.type.enumNativeName)
      );

    }

    funcDefinition.args.values().stream()
                       .filter(x -> x.type.type == ValueType.EnumRef)
                       .map(x -> generateEnumDefinition(x.enumFullClassName, x.type.enumNativeName))
                       .forEach(list::add);

    return list.stream();

  }

  private static Stream<EnumDefinition> toEnumDefinitions(ObjectMemberDefinition objectMemberDefinition) {

    List<EnumDefinition> list = new ArrayList<>();

    if (objectMemberDefinition.fieldType != null && objectMemberDefinition.fieldType.type == ValueType.EnumRef) {

      list.add(
        generateEnumDefinition(objectMemberDefinition.enumFullClassName,
                               objectMemberDefinition.fieldType.enumNativeName)
      );

    }

    if (objectMemberDefinition.methodRet != null && objectMemberDefinition.methodRet.valueType.type == ValueType.EnumRef) {

      list.add(
        generateEnumDefinition(objectMemberDefinition.methodRet.enumFullClassName,
                               objectMemberDefinition.methodRet.valueType.enumNativeName)
      );

    }

    objectMemberDefinition.methodArgs.values().stream()
                                     .filter(x -> x.type.type == ValueType.EnumRef)
                                     .map(x -> generateEnumDefinition(x.enumFullClassName, x.type.enumNativeName))
                                     .forEach(list::add);

    return list.stream();

  }

  @SneakyThrows
  private static EnumDefinition generateEnumDefinition(String enumClassName, String enumNativeName) {

    Class<?> enumClass = Class.forName(enumClassName);

    if (!enumClass.isEnum()) {
      throw new RuntimeException("yM8t6gcGs8 :: Класс " + enumClass.getSimpleName() + " не является енумом");
    }

    EnumName enumName = enumClass.getAnnotation(EnumName.class);
    if (enumName == null) {
      throw new RuntimeException("C1tH887pX5 :: Нет аннотации EnumName у енума " + enumClass.getSimpleName());
    }
    EnumDescription enumDescription = enumClass.getAnnotation(EnumDescription.class);
    if (enumDescription == null) {
      throw new RuntimeException("AT8Es2xO80 :: Нет аннотации EnumDescription у енума " + enumClass.getSimpleName());
    }

    EnumDefinition enumDefinition = new EnumDefinition();
    enumDefinition.name          = enumName.value();
    enumDefinition.description   = enumDescription.value();
    enumDefinition.nativeName    = enumNativeName;
    enumDefinition.fullClassName = enumClass.getName();
    enumDefinition.type          = ValueExtType.enumRef(enumNativeName);

    Arrays.stream((((Class<? extends Enum<?>>) enumClass).getEnumConstants()))
          .map(EnumsDefinitionExtractor::toEnumValueDefinition)
          .forEach(x -> enumDefinition.values.put(x.nativeName, x));

    return enumDefinition;

  }

  @SneakyThrows
  private static EnumValueDefinition toEnumValueDefinition(Enum<?> x) {

    EnumValueDefinition enumValueDefinition = new EnumValueDefinition();

    Class<?> enumClass = x.getDeclaringClass();

    var map = Arrays.stream(enumClass.getField(x.name()).getAnnotations()).collect(Collectors.toMap(Annotation::annotationType, v -> v));

    EnumName enumName = (EnumName) map.get(EnumName.class);
    if (enumName == null) {
      throw new RuntimeException("hUi4wM2c75 :: Нет аннотации EnumName у значения енума: " + enumClass.getSimpleName() + "." + x.name());
    }
    EnumDescription enumDescription = (EnumDescription) map.get(EnumDescription.class);
    if (enumDescription == null) {
      throw new RuntimeException("lClbVgb0kD :: Нет аннотации EnumDescription у значения енума: " + enumClass.getSimpleName() + "." + x.name());
    }

    enumValueDefinition.name        = enumName.value();
    enumValueDefinition.description = enumDescription.value();
    enumValueDefinition.nativeName  = x.name();

    return enumValueDefinition;

  }

}
