package kz.greetgo.script.model.definitions.object.extractor;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import kz.greetgo.script.ann.object.ObjectConstructor;
import kz.greetgo.script.ann.object.ObjectCreateMethodName;
import kz.greetgo.script.ann.object.ObjectDescription;
import kz.greetgo.script.ann.object.ObjectDisabled;
import kz.greetgo.script.ann.object.ObjectFieldType;
import kz.greetgo.script.ann.object.ObjectMethodParamDescription;
import kz.greetgo.script.ann.object.ObjectMethodParamName;
import kz.greetgo.script.ann.object.ObjectMethodParamType;
import kz.greetgo.script.ann.object.ObjectMethodReturnDescription;
import kz.greetgo.script.ann.object.ObjectMethodReturnType;
import kz.greetgo.script.ann.object.ObjectName;
import kz.greetgo.script.ann.object.ObjectNativeName;
import kz.greetgo.script.ann.object.PersonRef;
import kz.greetgo.script.model.context.ScriptObjectContext;
import kz.greetgo.script.model.context.model.BoiRef;
import kz.greetgo.script.model.context.model.signature.MybpmFile;
import kz.greetgo.script.model.definitions.object.MethodRetDefinition;
import kz.greetgo.script.model.definitions.object.ObjectDefinition;
import kz.greetgo.script.model.definitions.object.ObjectMemberDefinition;
import kz.greetgo.script.model.definitions.object.ObjectMemberType;
import kz.greetgo.script.model.definitions.object.ObjectMethodArgDefinition;
import kz.greetgo.script.model.definitions.object.ObjectsDefinition;
import kz.greetgo.script.model.translate.ValueExtType;
import kz.greetgo.script.model.util.ValueExtTypeUtil;
import org.jetbrains.annotations.Nullable;

import static java.util.stream.Collectors.toMap;

public class ObjectsDefinitionExtractor {

  private static final Map<Class<? extends ScriptObjectContext>, ObjectsDefinition> cached = new ConcurrentHashMap<>();

  public static void init(Class<? extends ScriptObjectContext> scriptObjectContextClass) {
    loadObjectsDefinition(scriptObjectContextClass);
  }

  public static ObjectsDefinition loadObjectsDefinition(Class<? extends ScriptObjectContext> scriptObjectContextClass) {

    {
      ObjectsDefinition fromCache = cached.get(scriptObjectContextClass);
      if (fromCache != null) {
        return fromCache;
      }
    }

    synchronized (cached) {
      ObjectsDefinition fromCache = cached.get(scriptObjectContextClass);
      if (fromCache != null) {
        return fromCache;
      }

      ObjectsDefinition load = load(scriptObjectContextClass);

      cached.put(scriptObjectContextClass, load);

      return load;
    }

  }

  @Nullable
  public static ObjectsDefinition loadFirstObjectsDefinition() {
    synchronized (cached) {
      return cached.values().stream()
                   .findFirst()
                   .orElse(null);
    }
  }

  private static ObjectsDefinition load(Class<? extends ScriptObjectContext> scriptObjectContextClass) {

    ObjectsDefinition objectsDefinition = new ObjectsDefinition();

    objectsDefinition.objects = Arrays.stream(scriptObjectContextClass.getDeclaredMethods())
                                      .filter(x -> x.isAnnotationPresent(ObjectConstructor.class))
                                      .map(ObjectsDefinitionExtractor::extract)
                                      .collect(Collectors.toMap(k -> k.nativeName, v -> v));

    return objectsDefinition;

  }

  private static ObjectDefinition extract(Method method) {

    ObjectDefinition objectDefinition = new ObjectDefinition();

    Class<?> returnType = method.getReturnType();

    ObjectNativeName objectNativeName = returnType.getAnnotation(ObjectNativeName.class);
    if (objectNativeName == null) {
      throw new RuntimeException("Нет аннотации ObjectNativeName у класса " + returnType.getName());
    }
    ObjectName objectName = returnType.getAnnotation(ObjectName.class);
    if (objectName == null) {
      throw new RuntimeException("Нет аннотации ObjectName у класса " + returnType.getName());
    }
    ObjectDescription objectDescription = returnType.getAnnotation(ObjectDescription.class);
    if (objectDescription == null) {
      throw new RuntimeException("Нет аннотации ObjectDescription у класса " + returnType.getName());
    }
    ObjectCreateMethodName objectCreateMethodName = returnType.getAnnotation(ObjectCreateMethodName.class);
    String                 createMethodName;
    if (objectCreateMethodName != null) {
      createMethodName = objectCreateMethodName.value();
    } else {
      createMethodName = "create" + objectNativeName.value();
    }
    boolean disabled = method.isAnnotationPresent(ObjectDisabled.class);

    objectDefinition.name             = objectName.value();
    objectDefinition.description      = objectDescription.value();
    objectDefinition.nativeName       = objectNativeName.value();
    objectDefinition.disabled         = disabled;
    objectDefinition.createMethodName = createMethodName;
    objectDefinition.isFile           = MybpmFile.class.equals(returnType);

    Map<String, ObjectMemberDefinition> members = Arrays.stream(returnType.getDeclaredFields())
                                                        .filter(x -> Modifier.isPublic(x.getModifiers()))
                                                        .map(x -> toObjectMemberDefinition(x, returnType.getSimpleName()))
                                                        .filter(Objects::nonNull)
                                                        .collect(Collectors.toMap(k -> k.nativeName, v -> v));

    Arrays.stream(returnType.getDeclaredMethods())
          .filter(x -> Modifier.isPublic(x.getModifiers()))
          .map(x -> toObjectMemberDefinition(x, returnType.getSimpleName()))
          .forEach(x -> members.put(x.nativeName, x));

    objectDefinition.members = members;

    return objectDefinition;

  }

  private static ObjectMemberDefinition toObjectMemberDefinition(Field field, String classSimpleName) {

    ObjectMemberDefinition objectMemberDefinition = new ObjectMemberDefinition();
    String                 fieldPath              = classSimpleName + "." + field.getName();

    ObjectName objectName = field.getAnnotation(ObjectName.class);
    if (objectName == null) {
      return null;
    }
    ObjectDescription objectDescription = field.getAnnotation(ObjectDescription.class);
    if (objectDescription == null) {
      throw new RuntimeException("Нет аннотации ObjectDescription у поля " + fieldPath);
    }

    Class<?> type = field.getType();

    objectMemberDefinition.name              = objectName.value();
    objectMemberDefinition.description       = objectDescription.value();
    objectMemberDefinition.nativeName        = field.getName();
    objectMemberDefinition.enumFullClassName = type.isEnum() ? type.getName() : null;
    objectMemberDefinition.memberType        = ObjectMemberType.FIELD;
    objectMemberDefinition.fieldType         = toValueExtTypeFromField(field, fieldPath);
    objectMemberDefinition.fieldReadonly     = Modifier.isFinal(field.getModifiers());
    objectMemberDefinition.isPersonBo        = objectMemberDefinition.fieldType.isPersonBo;

    return objectMemberDefinition;

  }

  private static ObjectMemberDefinition toObjectMemberDefinition(Method method, String classSimpleName) {

    ObjectMemberDefinition objectMemberDefinition = new ObjectMemberDefinition();

    String methodPath = classSimpleName + "." + method.getName();

    ObjectName objectName = method.getAnnotation(ObjectName.class);
    if (objectName == null) {
      throw new RuntimeException("4e0Gg03AbL Нет аннотации ObjectName у метода " + methodPath);
    }
    ObjectDescription objectDescription = method.getAnnotation(ObjectDescription.class);
    if (objectDescription == null) {
      throw new RuntimeException("OiiMl75WcG Нет аннотации ObjectDescription у метода " + methodPath);
    }

    objectMemberDefinition.name        = objectName.value();
    objectMemberDefinition.description = objectDescription.value();
    objectMemberDefinition.nativeName  = method.getName();
    objectMemberDefinition.memberType  = ObjectMemberType.METHOD;

    if (method.getGenericReturnType() != Void.TYPE) {

      ObjectMethodReturnDescription objectMethodReturnDescription = method.getAnnotation(ObjectMethodReturnDescription.class);
      if (objectMethodReturnDescription == null) {
        throw new RuntimeException("193XO206V5 :: Нет аннотации ObjectMethodReturnDescription у возвращаемого элемента, метода " + methodPath);
      }

      Class<?> returnType = method.getReturnType();

      MethodRetDefinition methodRetDefinition = new MethodRetDefinition();
      methodRetDefinition.description       = objectMethodReturnDescription.value();
      methodRetDefinition.enumFullClassName = returnType.isEnum() ? returnType.getName() : null;
      methodRetDefinition.valueType         = toValueExtTypeFromReturnType(method, methodPath);

      objectMemberDefinition.methodRet = methodRetDefinition;

    }

    Annotation[][] parameterAnnotations = method.getParameterAnnotations();

    for (int i = 0; i < method.getParameterCount(); i++) {
      objectMemberDefinition.methodArgs.put("a" + i, toObjectMethodArgDefinition(method, i, parameterAnnotations[i], methodPath));
    }

    return objectMemberDefinition;

  }

  private static ObjectMethodArgDefinition toObjectMethodArgDefinition(Method method, int parameterIndex, Annotation[] parameterAnnotations,
                                                                       String methodPath) {

    ObjectMethodArgDefinition objectMethodArgDefinition = new ObjectMethodArgDefinition();

    var map = Arrays.stream(parameterAnnotations).collect(toMap(Annotation::annotationType, v -> v));

    ObjectMethodParamName parameterName = (ObjectMethodParamName) map.get(ObjectMethodParamName.class);
    if (parameterName == null) {
      throw new RuntimeException("9W7250HK1N :: нет аннотации ObjectMethodParamName у параметра с индексом " + parameterIndex
                                   + " у метода " + methodPath);
    }
    ObjectMethodParamDescription parameterDescription = (ObjectMethodParamDescription) map.get(ObjectMethodParamDescription.class);
    if (parameterDescription == null) {
      throw new RuntimeException("2RVFWhgQJJ :: нет аннотации ObjectMethodParamDescription у параметра с индексом " + parameterIndex
                                   + " у метода " + methodPath);
    }

    Parameter parameter = method.getParameters()[parameterIndex];

    Class<?> type = parameter.getType();

    objectMethodArgDefinition.name              = parameterName.value();
    objectMethodArgDefinition.enumFullClassName = type.isEnum() ? type.getName() : null;
    objectMethodArgDefinition.description       = parameterDescription.value();
    objectMethodArgDefinition.nativeIndex       = parameterIndex;
    objectMethodArgDefinition.type              = toValueExtTypeFromMethodParameter(parameter, methodPath);

    return objectMethodArgDefinition;

  }

  private static ValueExtType toValueExtTypeFromField(Field field, String path) {

    Class<?> fieldClass = field.getType();

    ValueExtType valueExtType = ValueExtTypeUtil.fromSimpleToValueExtType(fieldClass);
    if (valueExtType != null) {
      return valueExtType;
    }

    valueExtType = getValueExtTypeFromObjects(fieldClass);
    if (valueExtType != null) {
      return valueExtType;
    }

    Map<? extends Class<? extends Annotation>, Annotation> annotationMap = Arrays.stream(field.getAnnotations()).collect(
      toMap(Annotation::annotationType, v -> v));

    if (Objects.equals(BoiRef.class, fieldClass)) {
      ObjectFieldType objectFieldType = (ObjectFieldType) annotationMap.get(ObjectFieldType.class);
      if (objectFieldType == null) {
        throw new RuntimeException("lCvrTBL1su :: нет аннотации ObjectFieldType у возвращаемого элемента для класса " +
                                     fieldClass.getSimpleName() + " поля: " + path);
      }

      ValueExtType extType = ValueExtType.single(objectFieldType.value());

      if (annotationMap.containsKey(PersonRef.class)) {
        extType.isPersonBo = true;
      }

      return extType;

    }

    if (List.class.equals(fieldClass)) {

      Type[] types = ((ParameterizedType) (field.getAnnotatedType().getType())).getActualTypeArguments();

      if (types.length != 1) {
        throw new RuntimeException("5O8nGkJU4z :: Невозможно прочитать generic class List-а");
      }

      Type type = types[0];

      if (!(type instanceof Class)) {
        throw new RuntimeException("dwv0s30AaE :: Cannot cast type into class");
      }

      Class<?> innerGenericClass = (Class<?>) type;

      if (BoiRef.class.equals(innerGenericClass)) {
        ObjectFieldType objectFieldType = (ObjectFieldType) annotationMap.get(ObjectFieldType.class);
        if (objectFieldType == null) {
          throw new RuntimeException(
            "hL9oOIA8Tb :: нет аннотации ObjectFieldType у возвращаемого элемента для класса " + innerGenericClass.getSimpleName() + ", метод: " + path);
        }
        return ValueExtType.multiple(objectFieldType.value());
      }

      ValueExtType innerValueExtType = ValueExtTypeUtil.fromSimpleToValueExtType(innerGenericClass);
      if (innerValueExtType != null) {
        innerValueExtType.isArray = true;
        return innerValueExtType;
      }

      innerValueExtType = getValueExtTypeFromObjects(innerGenericClass);
      if (innerValueExtType != null) {
        innerValueExtType.isArray = true;
        return innerValueExtType;
      }

    }

    throw new RuntimeException("Td1B37D352 :: Отсутствует обработчик для класса: " + fieldClass.getSimpleName() + ", путь: " + path);
  }

  private static ValueExtType toValueExtTypeFromReturnType(Method sourceMethod, String path) {

    Class<?> methodReturnClass = sourceMethod.getReturnType();

    ValueExtType valueExtType = ValueExtTypeUtil.fromSimpleToValueExtType(methodReturnClass);
    if (valueExtType != null) {
      return valueExtType;
    }

    valueExtType = getValueExtTypeFromObjects(methodReturnClass);
    if (valueExtType != null) {
      return valueExtType;
    }

    Annotation[] annotations = sourceMethod.getAnnotatedReturnType().getAnnotations();

    Map<? extends Class<? extends Annotation>, Annotation> annotationMap =
      Arrays.stream(annotations).collect(toMap(Annotation::annotationType, v -> v));

    if (Objects.equals(BoiRef.class, methodReturnClass)) {
      ObjectMethodReturnType objectMethodReturnType = (ObjectMethodReturnType) annotationMap.get(ObjectMethodReturnType.class);
      if (objectMethodReturnType == null) {
        throw new RuntimeException(
          "V9XWu7BaIS :: нет аннотации ObjectMethodReturnType у возвращаемого элемента для класса " + methodReturnClass.getSimpleName()
            + ", метод: " + path);
      }
      return ValueExtType.single(objectMethodReturnType.value());
    }

    if (List.class.equals(methodReturnClass)) {

      Type[] types = ((ParameterizedType) (sourceMethod.getAnnotatedReturnType().getType())).getActualTypeArguments();

      if (types.length != 1) {
        throw new RuntimeException("XA8HTPYvHa :: Невозможно прочитать generic class List-а");
      }

      Type type = types[0];

      if (!(type instanceof Class)) {
        throw new RuntimeException("HF8YUP068s :: Cannot cast type into class");
      }

      Class<?> innerGenericClass = (Class<?>) type;

      if (BoiRef.class.equals(innerGenericClass)) {
        ObjectMethodReturnType objectMethodReturnType = (ObjectMethodReturnType) annotationMap.get(ObjectMethodReturnType.class);
        if (objectMethodReturnType == null) {
          throw new RuntimeException(
            "B0Eq8KO7tn :: нет аннотации ObjectMethodReturnType у возвращаемого элемента для класса " + methodReturnClass.getSimpleName()
              + ", метод: " + path);
        }
        return ValueExtType.multiple(objectMethodReturnType.value());
      }

      ValueExtType innerValueExtType = ValueExtTypeUtil.fromSimpleToValueExtType(innerGenericClass);
      if (innerValueExtType != null) {
        innerValueExtType.isArray = true;
        return innerValueExtType;
      }

      innerValueExtType = getValueExtTypeFromObjects(innerGenericClass);
      if (innerValueExtType != null) {
        innerValueExtType.isArray = true;
        return innerValueExtType;
      }

    }

    throw new RuntimeException("GY1oXDw7Hf :: Отсутствует обработчик для класса : " + methodReturnClass.getSimpleName() + ", путь: " + path);
  }

  private static ValueExtType toValueExtTypeFromMethodParameter(Parameter sourceParameter,
                                                                String path) {

    Class<?> methodParameterClass = sourceParameter.getType();

    ValueExtType valueExtType = ValueExtTypeUtil.fromSimpleToValueExtType(methodParameterClass);
    if (valueExtType != null) {
      return valueExtType;
    }

    valueExtType = getValueExtTypeFromObjects(methodParameterClass);
    if (valueExtType != null) {
      return valueExtType;
    }

    Annotation[] annotations = sourceParameter.getAnnotations();

    Map<? extends Class<? extends Annotation>, Annotation> annotationMap = Arrays.stream(annotations).collect(
      toMap(Annotation::annotationType, v -> v));

    if (Objects.equals(BoiRef.class, methodParameterClass)) {
      ObjectMethodParamType objectMethodParamType = (ObjectMethodParamType) annotationMap.get(ObjectMethodParamType.class);
      if (objectMethodParamType == null) {
        throw new RuntimeException("6ytKN2VIj6 :: нет аннотации ObjectMethodParamType у параметра для класса " + methodParameterClass.getSimpleName()
                                     + ", параметр: " + path);
      }
      return ValueExtType.single(objectMethodParamType.value());
    }

    if (List.class.equals(methodParameterClass)) {

      Type[] types = ((ParameterizedType) (sourceParameter.getParameterizedType())).getActualTypeArguments();

      if (types.length != 1) {
        throw new RuntimeException("JyUx0ivD4U :: Невозможно прочитать generic class List-а");
      }

      Type type = types[0];

      if (!(type instanceof Class)) {
        throw new RuntimeException("7W54sk76ta :: Cannot cast type into class");
      }

      Class<?> innerGenericClass = (Class<?>) type;

      if (BoiRef.class.equals(innerGenericClass)) {
        ObjectMethodParamType objectMethodParamType = (ObjectMethodParamType) annotationMap.get(ObjectMethodParamType.class);
        if (objectMethodParamType == null) {
          throw new RuntimeException(
            "uhEW0I61KC :: нет аннотации ObjectMethodParamType у возвращаемого элемента для класса " + methodParameterClass.getSimpleName()
              + ", метод: " + path);
        }
        return ValueExtType.multiple(objectMethodParamType.value());
      }

      ValueExtType innerValueExtType = ValueExtTypeUtil.fromSimpleToValueExtType(innerGenericClass);
      if (innerValueExtType != null) {
        innerValueExtType.isArray = true;
        return innerValueExtType;
      }

      innerValueExtType = getValueExtTypeFromObjects(innerGenericClass);
      if (innerValueExtType != null) {
        innerValueExtType.isArray = true;
        return innerValueExtType;
      }

    }

    throw new RuntimeException("NH6jsMXauN :: Отсутствует обработчик для класса : " + methodParameterClass.getSimpleName() + ", путь: " + path);
  }

  @Nullable
  private static ValueExtType getValueExtTypeFromObjects(Class<?> aClass) {

    ObjectNativeName objectNativeName = aClass.getAnnotation(ObjectNativeName.class);
    if (objectNativeName == null) {
      return null;
    }

    return ValueExtType.object(objectNativeName.value());
  }

}
