package kz.greetgo.spring.websocket.mongo;

import com.mongodb.MongoException;
import com.mongodb.client.MongoChangeStreamCursor;
import com.mongodb.client.model.changestream.ChangeStreamDocument;
import kz.greetgo.spring.websocket.interfaces.NeedClose;
import kz.greetgo.spring.websocket.util.ConsoleColors;
import kz.greetgo.spring.websocket.util.LoggingUtil;
import org.bson.BsonDocument;
import org.bson.Document;
import org.slf4j.Logger;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

public abstract class AbstractMongoWatcher<DataObject> implements NeedClose {
  private final Logger callingLog = LoggingUtil.callingLog;

  public final String sessionId;
  private final Thread watchingThread;
  private final AtomicBoolean opened = new AtomicBoolean(true);

  private final AtomicReference<BsonDocument> resumeToken = new AtomicReference<>(null);

  public AbstractMongoWatcher(String sessionId) {
    this.sessionId = sessionId;
    watchingThread = new Thread(this::watch);
  }

  protected void start() {
    watchingThread.start();
    if (callingLog.isInfoEnabled()) {
      callingLog.info(ConsoleColors.CYAN_BOLD() + "WATCH     " + ConsoleColors.RESET()
        + sessionId
        + ConsoleColors.CYAN_BOLD() + ' ' + inWatchPlace() + ConsoleColors.RESET()
        + " Thread=" + Thread.currentThread().getName());
    }
  }

  @Override
  public void close() {
    opened.set(false);
    watchingThread.interrupt();
    if (callingLog.isInfoEnabled()) {
      callingLog.info(ConsoleColors.CYAN_BOLD() + "UNWATCH   " + ConsoleColors.RESET()
        + sessionId
        + ConsoleColors.CYAN_BOLD() + ' ' + inWatchPlace() + ConsoleColors.RESET()
        + " Thread=" + Thread.currentThread().getName()
      );
    }
  }

  protected abstract String inWatchPlace();

  protected abstract MongoChangeStreamCursor<ChangeStreamDocument<DataObject>> createCursor(BsonDocument resumeToken);

  protected abstract void acceptDoc(ChangeStreamDocument<DataObject> doc);

  private void watch() {
    if (callingLog.isInfoEnabled()) {
      callingLog.info("IN WATCH  "
        + sessionId
        + ConsoleColors.CYAN_BOLD() + ' ' + inWatchPlace() + ConsoleColors.RESET()
        + " Thread=" + Thread.currentThread().getName());
    }

    while (opened.get()) {
      try {
        var cursorRef = new AtomicReference<MongoChangeStreamCursor<ChangeStreamDocument<DataObject>>>(null);

        try (MongoChangeStreamCursor<ChangeStreamDocument<DataObject>> cursor = createCursor(resumeToken.get())) {
          cursorRef.set(cursor);

          if (callingLog.isInfoEnabled()) {
            callingLog.info("NEW CUR   "
              + sessionId
              + ConsoleColors.GREEN_BOLD() + " cursor.addr=" + System.identityHashCode(cursor) + ConsoleColors.RESET()
              + " Thread=" + Thread.currentThread().getName());
          }

          cursor.forEachRemaining(doc -> {
            resumeToken.set(doc.getResumeToken());
            acceptDoc(doc);
          });
        } catch (MongoException e) {
          MongoErrorFilters.mongo(e)
            .ignoreStateShouldBeOpened(true)
            .ignoreCursorHasBeenClosed(true)
            .ignoreMongoInterrupted(true)
            .check();
        } finally {
          var cc = cursorRef.get();
          if (cc != null) {
            if (callingLog.isInfoEnabled()) {
              callingLog.info("CLO CUR   "
                + sessionId
                + ConsoleColors.RED_BOLD() + " cursor.addr=" + System.identityHashCode(cc) + ConsoleColors.RESET()
                + " Thread=" + Thread.currentThread().getName());
            }
          }
        }
      } catch (Exception e) {
        callingLog.error(ConsoleColors.RED_BOLD() + "ERROR    " + ConsoleColors.RESET()
            + ' ' + sessionId + ' '
            + ConsoleColors.RED() + " Ошибка в смотрителе: "
            + e.getClass().getName() + " : " + e.getMessage()
            + ConsoleColors.RESET(),
          e);
      }
    }

    if (callingLog.isInfoEnabled()) {
      callingLog.info("NO WATCH  "
        + sessionId
        + ConsoleColors.CYAN_BOLD() + ' ' + inWatchPlace() + ConsoleColors.RESET()
        + " Thread=" + Thread.currentThread().getName());
    }
  }

  public void join() {
    try {
      watchingThread.join();
    } catch (InterruptedException e) {
      //ignore
    }
  }

}
