package kz.greetgo.spring.websocket.controller;

import kz.greetgo.spring.websocket.interfaces.ServiceExecutor;
import kz.greetgo.spring.websocket.util.ConsoleColors;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

public class ControllerManager {

  Logger logger = LoggerFactory.getLogger(getClass());

  private final ConcurrentHashMap<String, ServiceExecutor> serviceExecutorMap = new ConcurrentHashMap<>();

  private Supplier<PreExecuteInterceptor> preExecuteInterceptorSupplier = () -> null;

  public void setPreExecuteInterceptorSupplier(@NonNull Supplier<PreExecuteInterceptor> preExecuteInterceptorSupplier) {
    this.preExecuteInterceptorSupplier = requireNonNull(preExecuteInterceptorSupplier);
  }

  @SuppressWarnings("UnusedReturnValue")
  public Object findAndExecuteService(String serviceName, ExecuteInput executeInput) {

    ServiceExecutor serviceExecutor = serviceExecutorMap.get(serviceName);

    if (serviceExecutor != null) {
      return serviceExecutor.execute(executeInput);
    }

    logger.error("XliQsOf6Ad :: No service with name " + serviceName);
    return null;
  }

  public void registerController(Object controller) {
    ControllerParser controllerParser = new ControllerParser();
    controllerParser.setPreExecuteInterceptorSupplier(preExecuteInterceptorSupplier);

    List<ServiceExecutor> serviceExecutorList = controllerParser.parse(controller);

    for (ServiceExecutor serviceExecutor : serviceExecutorList) {

      ServiceExecutor exists = serviceExecutorMap.get(serviceExecutor.serviceName());
      if (exists != null) {
        throw new RuntimeException("Cjk6xs4XvY :: You try to register two services with"
                                     + " same name: " + serviceExecutor.serviceName()
                                     + "\n\tService 1 placed at " + exists.placeDisplayStr()
                                     + "\n\tService 2 placed at " + serviceExecutor.placeDisplayStr());
      }

      serviceExecutorMap.put(serviceExecutor.serviceName(), serviceExecutor);

      if (logger.isInfoEnabled()) {
        logger.info(""
                      + ConsoleColors.PURPLE_BOLD() + "ws   controller" + ConsoleColors.RESET() + " "
                      + ConsoleColors.BLUE() + serviceExecutor.serviceName() + ConsoleColors.RESET()
                      + ConsoleColors.GREEN_BOLD() + " -> " + ConsoleColors.RESET()
                      + ConsoleColors.BLACK_BOLD() + serviceExecutor.placeDisplayStr() + ConsoleColors.RESET());
      }

    }
  }
}
