package kz.greetgo.spring.websocket.controller;

import kz.greetgo.spring.websocket.exceptions.CannotCreateParamExtractor;
import kz.greetgo.spring.websocket.interfaces.Prm;
import kz.greetgo.spring.websocket.interfaces.ServiceExecutor;
import kz.greetgo.spring.websocket.interfaces.ServiceName;
import kz.greetgo.spring.websocket.interfaces.SessionId;
import kz.greetgo.spring.websocket.interfaces.WebsocketController;
import kz.greetgo.spring.websocket.util.MethodParamUtil;
import lombok.NonNull;
import org.springframework.core.annotation.AnnotationUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

public class ControllerParser {

  private Supplier<PreExecuteInterceptor> preExecuteInterceptor = () -> null;

  public void setPreExecuteInterceptorSupplier(@NonNull Supplier<PreExecuteInterceptor> preExecuteInterceptor) {
    this.preExecuteInterceptor = requireNonNull(preExecuteInterceptor);
  }

  public List<ServiceExecutor> parse(Object controller) {

    WebsocketController websocketController = AnnotationUtils.findAnnotation(
      controller.getClass(), WebsocketController.class);

    if (websocketController == null) {
      return Collections.emptyList();
    }

    List<ServiceExecutor> ret = new ArrayList<>();

    for (Method method : controller.getClass().getMethods()) {

      ServiceName serviceName = AnnotationUtils.findAnnotation(method, ServiceName.class);
      if (serviceName == null) {
        continue;
      }

      String serviceFullName = websocketController.value() + "/" + serviceName.value();

      ret.add(createExecutorOn(controller, method, serviceFullName));

    }

    return Collections.unmodifiableList(ret);

  }

  interface ParamExtractor {
    Object extract(ExecuteInput executeInput);
  }

  private ServiceExecutor createExecutorOn(Object controller, Method method, String serviceFullName) {

    Class<?>[] parameterTypes = method.getParameterTypes();
    Type[] genericParameterTypes = method.getGenericParameterTypes();
    Annotation[][] parameterAnnotations = MethodParamUtil.getAllParameterAnnotations(method);
    int parameterCount = method.getParameterCount();

    ParamExtractor[] paramExtractors = new ParamExtractor[parameterCount];
    for (int i = 0; i < parameterCount; i++) {
      paramExtractors[i] = createParamExtractor(
        parameterTypes[i],
        genericParameterTypes[i],
        parameterAnnotations[i],
        method);
    }

    Class<?> controllerClass = controller.getClass();

    AnnotationFinder annotationFinder = new AnnotationFinder() {
      @Override
      public <A extends Annotation> A find(Class<A> annotationType,
                                           boolean findOverMethod, boolean findOverControllerClass) {

        if (annotationType == null) {
          return null;
        }

        if (findOverMethod) {
          A annotation = AnnotationUtils.findAnnotation(method, annotationType);
          if (annotation != null) {
            return annotation;
          }
        }

        if (findOverControllerClass) {
          A annotation = AnnotationUtils.findAnnotation(controllerClass, annotationType);
          if (annotation != null) {
            return annotation;
          }
        }

        return null;
      }
    };

    return new ServiceExecutor() {
      @Override
      public String serviceName() {
        return serviceFullName;
      }

      @Override
      public String placeDisplayStr() {
        String simpleName = controller.getClass().getSimpleName();
        {
          int idx = simpleName.indexOf("$$");
          if (idx >= 0) {
            simpleName = simpleName.substring(0, idx);
          }
        }
        return simpleName + "." + method.getName() + "()";
      }

      @Override
      public Object execute(ExecuteInput executeInput) {

        {
          var x = preExecuteInterceptor.get();
          if (x != null) {
            x.preExecute(controller, method, serviceFullName, executeInput, annotationFinder);
          }
        }

        Object[] paramValues = new Object[parameterCount];
        for (int i = 0; i < parameterCount; i++) {
          paramValues[i] = paramExtractors[i].extract(executeInput);
        }

        try {
          return method.invoke(controller, paramValues);
        } catch (IllegalAccessException e) {
          throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
          Throwable cause = e.getCause();
          if (cause instanceof RuntimeException) {
            throw (RuntimeException) cause;
          }
          throw new RuntimeException("I68J6ww2H8 :: error in controller " + placeDisplayStr(), cause);
        }

      }
    };

  }

  private ParamExtractor createParamExtractor(Class<?> parameterType,
                                              Type genericParameterType,
                                              @NonNull Annotation[] parameterAnnotations,
                                              Method method) {

    for (Annotation parameterAnnotation : parameterAnnotations) {
      if (parameterAnnotation instanceof SessionId) {
        return executeSource -> executeSource.sessionIdSupplier.get();
      }

      if (parameterAnnotation instanceof Prm) {
        String prmName = ((Prm) parameterAnnotation).value();
        return executeSource -> executeSource.getParamConvertedToType(prmName, parameterType, genericParameterType);
      }
    }

    throw new CannotCreateParamExtractor(parameterType, genericParameterType, parameterAnnotations, method);
  }
}
