package li.rudin.arduino.managed.cdi;

import java.util.ArrayList;
import java.util.List;

import javax.enterprise.event.Observes;
import javax.enterprise.inject.spi.AfterBeanDiscovery;
import javax.enterprise.inject.spi.AnnotatedType;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.Extension;
import javax.enterprise.inject.spi.ProcessAnnotatedType;
import javax.enterprise.inject.spi.WithAnnotations;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import li.rudin.arduino.api.cdi.Connection;
import li.rudin.arduino.core.cache.TimedKeyValueCache;
import li.rudin.arduino.core.ethernet.ArduinoEthernetImpl;
import li.rudin.arduino.core.serial.ArduinoSerialImpl;

public class ManagedExtension implements Extension
{
	/**
	 * logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(ManagedExtension.class);

	<X> void processAnnotatedType(@Observes @WithAnnotations({Connection.class}) ProcessAnnotatedType<X> pat)
	{
		AnnotatedType<X> type = pat.getAnnotatedType();
		if (type.getAnnotation(Connection.class) != null)
		{
			logger.info("Found managed device class: '{}'", type.getJavaClass().getName());
			deviceList.add(type.getJavaClass());
		}
	}
	
	/**
	 * All found devices
	 */
	private final List<Class<?>> deviceList = new ArrayList<>();
	
	void afterBeanDiscovery(@Observes AfterBeanDiscovery abd, BeanManager bm)
	{
		try
		{
			for (Class<?> type: deviceList)
			{
				Connection connection = type.getAnnotation(Connection.class);

				if (connection != null)
				{
					if (connection.host().length() > 0)
					{
						logger.info("Creating ethernet device instance for type '{}', host: '{}' and port: '{}'",
								type.getName(), connection.host(), connection.port());
	
						ArduinoEthernetImpl arduino = new ArduinoEthernetImpl(connection.host(), connection.port());
						arduino.setCache(new TimedKeyValueCache());
	
						abd.addBean(new ManagedDeviceBean(type, arduino, bm));
					}
					else if (connection.serialPort().length() > 0)
					{
						logger.info("Creating serial device instance for type: '{}' speed: '{}' and port: '{}'",
								type.getName(), connection.serialSpeed(), connection.serialPort());
						
						ArduinoSerialImpl arduino = new ArduinoSerialImpl(connection.serialPort(), connection.serialSpeed());
						arduino.setCache(new TimedKeyValueCache());

						abd.addBean(new ManagedDeviceBean(type, arduino, bm));
					}
				}
			}

		}
		catch (Exception e)
		{
			abd.addDefinitionError(e);
		}

	}

}
