package li.rudin.arduino.managed.cdi;

import java.lang.annotation.Annotation;

import javax.enterprise.inject.spi.BeanManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import li.rudin.arduino.api.cdi.DeviceEvent;
import li.rudin.arduino.api.cdi.Direction;
import li.rudin.arduino.api.cdi.StateChangeEvent;
import li.rudin.arduino.api.listener.ArduinoListener;
import li.rudin.arduino.api.state.ConnectionState;

public class EventDistributor implements ArduinoListener
{
	/**
	 * Local logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(EventDistributor.class);

	public EventDistributor(BeanManager bm, Class<?> device)
	{
		this.bm = bm;
		this.device = device;
	}

	private final BeanManager bm;

	private final Class<?> device;

	@Override
	public void onMessageReceived(String key, String value)
	{
		logger.debug("Incoming event: device({}) key({}) value({})", device.getName(), key, value);
		fireDynamic(key, value, Direction.INCOMING);
	}

	@Override
	public void onMessageTransmitted(String key, String value)
	{
		logger.debug("Outgoing event: device({}) key({}) value({})", device.getName(), key, value);
		fireDynamic(key, value, Direction.OUTGOING);
	}

	private void fireDynamic(String key, String value, Direction direction)
	{
		DeviceEventType annotationType = new DeviceEventType(device, key, direction);

		//Fire string event
		bm.fireEvent(value, annotationType);

		if (value.length() == 0)
			//No more events
			return;

		try
		{
			Integer i = Integer.parseInt(value);

			//Fire Integer event
			bm.fireEvent(i, annotationType);

			if (i.equals(1))
				//Fire true boolean event
				bm.fireEvent(true, annotationType);

			else if (i.equals(0))
				//Fire false boolean event
				bm.fireEvent(false, annotationType);
		}
		catch (NumberFormatException e) {/* ignored */}

		try
		{
			Double d = Double.parseDouble(value);

			//Fire Double event
			bm.fireEvent(d, annotationType);
		}
		catch (NumberFormatException e) {/* ignored */}

		try
		{
			Long l = Long.parseLong(value);

			//Fire Long event
			bm.fireEvent(l, annotationType);
		}
		catch (NumberFormatException e) {/* ignored */}

	}

	private ConnectionState previousState = ConnectionState.DISCONNECTED;

	@Override
	public void onStateChange(ConnectionState currentState, ConnectionState targetState)
	{
		if (previousState != currentState)
		{
			logger.debug("Distributing state change: current({}) previous({})", currentState, previousState);

			DeviceEventType annotationType = new DeviceEventType(device, "", Direction.INCOMING);

			bm.fireEvent(new StateChangeEvent(previousState, currentState), annotationType);

			previousState = currentState;
		}
	}

	public static class DeviceEventType implements DeviceEvent
	{

		public DeviceEventType(Class<?> device, String key, Direction direction)
		{
			this.device = device;
			this.direction = direction;
			this.key = key;
		}

		private final Class<?> device;

		private final String key;

		private final Direction direction;

		@Override
		public Class<? extends Annotation> annotationType()
		{
			return DeviceEvent.class;
		}

		@Override
		public Class<?> device()
		{
			return device;
		}

		@Override
		public String key()
		{
			return key;
		}

		@Override
		public Direction direction()
		{
			return direction;
		}

	}

}
