package li.rudin.arduino.managed.cdi;

import java.net.URL;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.enterprise.event.Observes;
import javax.enterprise.inject.spi.AfterBeanDiscovery;
import javax.enterprise.inject.spi.BeanManager;
import javax.enterprise.inject.spi.Extension;
import javax.servlet.annotation.WebListener;

import li.rudin.arduino.api.ethernet.Connection;
import li.rudin.arduino.core.cache.TimedKeyValueCache;
import li.rudin.arduino.core.ethernet.ArduinoEthernetImpl;

import org.reflections.Reflections;
import org.reflections.util.ClasspathHelper;
import org.reflections.util.ConfigurationBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@WebListener
public class ManagedExtension implements Extension
{

	//http://stackoverflow.com/questions/13576665/unit-test-using-the-reflections-google-library-fails-only-when-executed-by-maven
	public static Set<URL> effectiveClassPathUrls(ClassLoader... classLoaders) {
		return ClasspathHelper.forManifest(ClasspathHelper.forClassLoader(classLoaders));
	}

	static
	{
		ExecutorService pool = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());
		
		ConfigurationBuilder config = 
				ConfigurationBuilder
				.build(new Object[]{}) //nice...
				.setExecutorService(pool)
				.addUrls(effectiveClassPathUrls(ManagedExtension.class.getClassLoader()));
				

		
		Iterator<URL> it = config.getUrls().iterator();
	    
	    while (it.hasNext())
	    {
	    	URL url = it.next();
	    	
	    	//Filter non-jar files (shared objects in classpath :( )
	    	if (url.getFile().contains(".so."))
	    		it.remove();
	    }
		
	    reflections = new Reflections(config);
	    
	    //shutdown pool
	    pool.shutdown();
	}

	private static final Reflections reflections;

	/**
	 * logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(ManagedExtension.class);

	void afterBeanDiscovery(@Observes AfterBeanDiscovery abd, BeanManager bm)
	{
		try
		{
			logger.info("Searching for managable devices");

			Set<Class<?>> types = reflections.getTypesAnnotatedWith(Connection.class);


			for (Class<?> type: types)
			{
				Connection connection = type.getAnnotation(Connection.class);

				if (connection != null)
				{
					logger.info("Creating device instance for type '{}', host: '{}' and port: '{}'", type.getName(), connection.host(), connection.port());

					ArduinoEthernetImpl arduino = new ArduinoEthernetImpl(connection.host(), connection.port());
					arduino.setCache(new TimedKeyValueCache());

					abd.addBean(new ManagedDeviceBean(type, arduino));
				}
			}

		}
		catch (Exception e)
		{
			abd.addDefinitionError(e);
		}

	}

}
