package li.rudin.arduino.testsuite;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import li.rudin.arduino.api.message.Message;
import li.rudin.arduino.core.message.MessageTranslator;


public class ArduinoEthernetDeviceServer implements Runnable, Consumer<Message>
{
	private static final Logger logger = LoggerFactory.getLogger(ArduinoEthernetDeviceServer.class);

	public ArduinoEthernetDeviceServer(int port)
	{
		this.port = port;
	}

	private final Map<String, String> valueMap = new HashMap<>();
	
	private final int port;
	private boolean run = false;
	private ServerSocket serversocket;
	private final MessageTranslator translator = new MessageTranslator(this, false);

	private List<Client> clients = new CopyOnWriteArrayList<>();

	public void start() throws IOException
	{
		run = true;
		serversocket = new ServerSocket(port);
		new Thread(this).start();
	}

	public void stop() throws IOException
	{
		logger.debug("Stopping server @ port {}", port);
		
		run = false;
		serversocket.close();
		
		for (Client c: clients)
			c.stop();
		
		logger.debug("Stopped server @ port {}", port);
	}

	
	public void send(String key, String value) throws IOException
	{
		for (Client c: clients){
			try {
				c.send(key, value);
			} catch (Exception e){
				logger.error("send to client", e);
			}
		}
			
	}

	public void send(String raw) throws IOException
	{
		for (Client c: clients)
			c.send(raw);
	}

	@Override
	public void run()
	{
		while(run)
		{
			try
			{
				Socket socket = serversocket.accept();
				new Client(socket);
			}
			catch (Exception e)
			{
				return;
			}
		}
	}
	
	public Stack<Message> getRxMessages()
	{
		return rxMessages;
	}

	public Map<String, String> getValueMap()
	{
		return valueMap;
	}

	private final Stack<Message> rxMessages = new Stack<>();

	class Client implements Runnable
	{

		private final Logger logger = LoggerFactory.getLogger(Client.class);
		
		public Client(Socket socket) throws IOException
		{
			input = socket.getInputStream();
			output = socket.getOutputStream();
			this.socket = socket;

			clients.add(this);
			new Thread(this).start();
		}

		private final InputStream input;
		private final OutputStream output;
		private final Socket socket;

		public void send(String key, String value) throws IOException
		{
			output.write( new Message(key, value).getBytes() );
			output.flush();
		}
		
		public void send(String raw) throws IOException
		{
			output.write( raw.getBytes() );
			output.flush();
		}
		
		public void stop() throws IOException
		{
			socket.close();
		}

		@Override
		public void run()
		{
			byte[] buffer = new byte[1024];

			while(run)
			{
				try
				{
					int count = input.read(buffer);

					logger.debug("Read {} bytes", count);
					
					if (count < 0)
						return;
					
					String str = new String(buffer, 0, count);
					translator.accept(str);
					
					try {
						translator.run();
					} catch (Exception e){
						logger.error("Error in listener", e);
					}
				}
				catch (Exception e)
				{
					return;
				}
			}
		}

	}

	@Override
	public void accept(Message msg) {
		logger.debug("Got {}", msg);
		
		if (msg.value.length() == 0 && valueMap.containsKey(msg.key))
		{
			//Get message
			try {
				String value = valueMap.get(msg.key);
				logger.debug("Sending key: {}, value: '{}'", msg.key, value);
				
				send(msg.key, value);
			} catch (Exception e){
				logger.error("send()", e);;
			}
		}
		else
			rxMessages.push(msg);
		
	}


}
