package li.rudin.core.security.impl;

import java.io.Serializable;

import javax.annotation.Priority;
import javax.inject.Inject;
import javax.interceptor.AroundInvoke;
import javax.interceptor.Interceptor;
import javax.interceptor.InvocationContext;

import li.rudin.core.security.Restricted;
import li.rudin.core.security.User;
import li.rudin.core.security.exception.NotAllowedException;
import li.rudin.core.security.service.CurrentUser;

import org.slf4j.Logger;

@Interceptor
@Restricted
@Priority(100)
public class RestrictionInterceptor implements Serializable
{

	@Inject CurrentUser currentUser;
	@Inject Logger logger;

	@AroundInvoke
	public Object intercept(InvocationContext ctx) throws Exception
	{

		//Extract from method
		Restricted restricted = ctx.getMethod().getAnnotation(Restricted.class);
		if (restricted == null)
			//Extract from class
			restricted = ctx.getMethod().getDeclaringClass().getAnnotation(Restricted.class);

		User user = currentUser.get();

		if (user == null)
			//should not be possible...
			throw new IllegalArgumentException("not logged in");

		boolean authorized = false;

		for (String availableRole: user.getRoles())
			for (String requiredRole: restricted.roles())
				if (availableRole.equals(requiredRole))
				{
					//Matching roles found
					authorized = true;
					break;
				}

		if (!authorized)
			for (String requiredUsername: restricted.users())
				if (requiredUsername.equals(user.getName()))
				{
					//Matching username found
					authorized = true;
					break;
				}

		if (!authorized)
		{
			String accessPoint = ctx.getMethod().getDeclaringClass().getName() + "." + ctx.getMethod().getName();

			if (restricted.logViolations())
				logger.warn("User {} tried to access {} without success", user.getName(), accessPoint);

			throw new NotAllowedException(user, ctx.getMethod().getDeclaringClass(), ctx.getMethod());
		}

		//Proceed
		Object result = ctx.proceed();


		return result;

	}


}
