package li.rudin.core.security.impl;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.annotation.Priority;
import javax.enterprise.context.spi.CreationalContext;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeanManager;
import javax.inject.Inject;
import javax.interceptor.AroundInvoke;
import javax.interceptor.Interceptor;
import javax.interceptor.InvocationContext;

import li.rudin.core.security.Restricted;
import li.rudin.core.security.RestrictionHandler;
import li.rudin.core.security.exception.NotAllowedException;

import org.slf4j.Logger;

@Interceptor
@Restricted
@Priority(100)
public class RestrictionInterceptor implements Serializable
{

	@Inject Logger logger;
	@Inject BeanManager beanManager;

	@AroundInvoke
	public Object intercept(InvocationContext ctx) throws Exception
	{

		Method method = ctx.getMethod();
		Class<?> type = method.getDeclaringClass();

		//Extract annotation
		List<Restricted> list = getRestrictedAnnotation(method);

		Class<? extends RestrictionHandler> restrictionHandler = list.get(0).restrictionHandler();
		
		//get handler instance
		Set<Bean<?>> beans = beanManager.getBeans(restrictionHandler);

		for (Bean<?> bean: beans)
		{
			CreationalContext<?> createCreationalContext = beanManager.createCreationalContext(bean);
			Object object = beanManager.getReference(bean, restrictionHandler, createCreationalContext);

			if (object instanceof RestrictionHandler)
			{
				RestrictionHandler handler = (RestrictionHandler)object;

				for (Restricted restricted: list)
					if (!handler.isAllowed(restricted, type, method))
						//Not authorized
						throw new NotAllowedException(type, ctx.getMethod());
				
				//Authorized
				return ctx.proceed();
			}

		}

		throw new IllegalArgumentException("could not find RestrictionHandler");
	}

	/**
	 * Extracts the @Resticted annotation from the method
	 * @param method
	 * @return
	 * @throws SecurityException 
	 * @throws NoSuchMethodException 
	 */
	private List<Restricted> getRestrictedAnnotation(Method method)
	{
		List<Restricted> list = new ArrayList<>();

		Class<?> type = method.getDeclaringClass();

		//Extract from method
		Restricted restricted = method.getAnnotation(Restricted.class);
		if (restricted != null)
			list.add(restricted);

		//Extract from class
		restricted = type.getAnnotation(Restricted.class);
		if (restricted != null)
			list.add(restricted);

		//interfaces
		for (Class<?> iface: type.getInterfaces())
			getRestrictedAnnotationFromInterface(list, iface, method);


		return list;
	}

	private void getRestrictedAnnotationFromInterface(List<Restricted> list, Class<?> iface, Method method)
	{
		//From interface annotation
		Restricted restricted = iface.getAnnotation(Restricted.class);
		if (restricted != null)
			list.add(restricted);

		//Recurse
		for (Class<?> subInterfaces: iface.getInterfaces())
			getRestrictedAnnotationFromInterface(list, subInterfaces, method);
		
		try
		{
			Method ifaceMethod = iface.getMethod(method.getName(), method.getParameterTypes());

			restricted = ifaceMethod.getAnnotation(Restricted.class);
			if (restricted != null)
				list.add(restricted);
		}
		catch (Exception e)
		{
			//Nothing to do here...
		}

	}


}
