package li.rudin.core.db;

import java.util.Iterator;

import javax.annotation.PostConstruct;
import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.inject.Instance;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;

import li.rudin.core.cdi.eager.Eager;

import org.slf4j.Logger;

@ApplicationScoped
@Eager
public class DatabaseBoot
{

	@Inject Instance<AutoversionDatabaseConfig> configs;
	@Inject EntityManagerProducer emProducer;
	@Inject Logger logger;


	@PostConstruct
	public void init()
	{
		Iterator<AutoversionDatabaseConfig> it = configs.iterator();
		while(it.hasNext())
			init(it.next());
	}
	
	private void injectSchema(EntityManager em, String schemaUrl)
	{
		EntityTransaction tx = em.getTransaction();
		tx.begin();
		
		logger.info("Injecting script from url: {}", schemaUrl);
		em.createNativeQuery("runscript from 'classpath:"+schemaUrl+"';").executeUpdate();
		
		em.flush();
		
		tx.commit();

	}
	
	private void updateSchema(EntityManager em, AutoversionDatabaseConfig dbConfig, long version) throws Exception
	{		
	
		while(true)
		{
			String schemaUrl = dbConfig.getSchema(version);
			
			if (DatabaseBoot.class.getResourceAsStream(schemaUrl) == null)
				//No schema found
				break;
			
			logger.debug("Found schema with version: {}", version);
			
			injectSchema(em, schemaUrl);
			
			EntityTransaction tx = em.getTransaction();
			tx.begin();
			dbConfig.setSchemaVersion(em, version);
			tx.commit();
			
			logger.info("Updated database schema version: {}", version);
			
			//check next version
			version++;
		}
		

	}

	public void init(AutoversionDatabaseConfig dbConfig)
	{
		logger.info("Setting up embedded database: {}", dbConfig.getName());
		
		EntityManager em = emProducer.getOrCreateEntityManager(dbConfig.getName());
		
		long version = -1L;
		
		try
		{
			//Try to retrieve current schema version
			version = dbConfig.getSchemaVersion(em);
		}
		catch (Exception e)
		{
			logger.info("Schema versioning not present, defaulting to 0");
		}
		
		//Increment schema version
		version++;

		try
		{
			//Update to next schema version or 0L if initial
			updateSchema(em, dbConfig, version);
		}
		catch (Exception e)
		{
			throw new IllegalArgumentException("Could not update schema to version: " + version, e);
		}
	}
	

}
