package li.rudin.core.jooq.impl;

import java.sql.Connection;
import java.sql.SQLException;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.inject.Disposes;
import javax.enterprise.inject.Produces;
import javax.enterprise.inject.spi.InjectionPoint;
import javax.inject.Inject;

import li.rudin.core.jooq.api.DatabaseConfiguration;

import org.jooq.Configuration;
import org.jooq.ConnectionProvider;
import org.jooq.DSLContext;
import org.jooq.SQLDialect;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.slf4j.Logger;

import com.mchange.v2.c3p0.ComboPooledDataSource;

@ApplicationScoped
public class ConfigurationProducer implements ConnectionProvider
{

	@Inject Logger logger;
	@Inject DatabaseConfiguration dbConfig;

	@PostConstruct
	void init()
	{
		ctx = DSL.using(SQLDialect.H2);

		configuration = ctx.configuration();
		configuration.set(this);
		try
		{
			dataSource = new ComboPooledDataSource();
			dataSource.setDriverClass("org.h2.Driver");
			dataSource.setUser("sa");
			dataSource.setPassword("");
			dataSource.setJdbcUrl(dbConfig.getDatabaseUrl());

			dataSource.setMinPoolSize(5);
			dataSource.setAcquireIncrement(5);
			dataSource.setMaxPoolSize(50);

		}
		catch (Exception e)
		{
			logger.error("init", e);
			throw new IllegalArgumentException("init", e);
		}
	}

	private DSLContext ctx;
	private Configuration configuration;
	private ComboPooledDataSource dataSource;

	@PreDestroy
	void destroy()
	{
		dataSource.close();
	}
	
	@Produces
	public Configuration produceConfiguration(InjectionPoint p)
	{
		return configuration;
	}


	@Override
	public Connection acquire() throws DataAccessException
	{
		return produceConnection();
	}

	@Override
	public void release(Connection connection) throws DataAccessException
	{
		disposeConnection(connection);
	}
	
	@Produces
	public Connection produceConnection()
	{
		try
		{
			return dataSource.getConnection();
		}
		catch (SQLException e)
		{
			throw new IllegalArgumentException("acquire", e);
		}

	}
	
	public void disposeConnection(@Disposes Connection connection)
	{
		try
		{
			connection.close();
			logger.debug("closed connection: {}", connection.hashCode());
		}
		catch (SQLException e)
		{
			logger.error("release", e);
		}

	}


}
