package li.rudin.core.jpa;

import javax.annotation.PostConstruct;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;

import li.rudin.core.cdi.eager.Eager;
import li.rudin.core.jpa.version.VersionInterface;

import org.slf4j.Logger;

@ApplicationScoped
@Eager
public class DatabaseBoot
{

	@Inject AutoversionDatabaseConfig config;
	@Inject PersistenceConfiguration persistenceConfig;
	@Inject EntityManager em;
	@Inject Logger logger;


	@PostConstruct
	public void init()
	{
		try
		{
			init(config);
		}
		catch (Exception e)
		{
			logger.error("init", e);
			throw new IllegalArgumentException("init", e);
		}	
	}

	private void injectSchema(EntityManager em, String schemaUrl)
	{
		EntityTransaction tx = em.getTransaction();
		tx.begin();

		logger.info("Injecting script from url: {}", schemaUrl);
		em.createNativeQuery("runscript from 'classpath:"+schemaUrl+"';").executeUpdate();

		tx.commit();
	}

	private void updateSchema(EntityManager em, AutoversionDatabaseConfig dbConfig, VersionInterface version) throws Exception
	{		

		while(true)
		{
			String schemaUrl = dbConfig.getSchema(version.getVersion());

			if (DatabaseBoot.class.getResourceAsStream(schemaUrl) == null)
				//No schema found
				break;

			logger.debug("Found schema with version: {}", version);

			injectSchema(em, schemaUrl);

			logger.info("Updated database schema version: {}", version);

			//check next version
			version.setVersion( version.getVersion()+1 );
		}


	}

	public void init(AutoversionDatabaseConfig dbConfig) throws Exception
	{
		logger.info("Setting up embedded database: {}", persistenceConfig.getUnitName());

		Class<? extends VersionInterface> versionType = dbConfig.getVersionEntity();

		VersionInterface version = null;

		try
		{
			//Try to retrieve current schema version
			version = em.find(versionType, 0L);

			//Increment schema version
			version.setVersion( version.getVersion()+1 );
		}
		catch (Exception e)
		{
			logger.info("Schema versioning not present, defaulting to 0");
			version = versionType.newInstance();
		}

		try
		{
			//Update to next schema version or 0L if initial
			updateSchema(em, dbConfig, version);

			EntityTransaction tx = em.getTransaction();
			tx.begin();
			if (em.contains(version))
				em.merge(version);
			else
				em.persist(version);

			em.flush();
			tx.commit();
		}
		catch (Exception e)
		{
			throw new IllegalArgumentException("Could not update schema to version: " + version.getVersion(), e);
		}
	}


}
