package li.rudin.core.scheduler.extension;

import java.lang.annotation.Annotation;

import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.event.Observes;
import javax.enterprise.inject.spi.AfterDeploymentValidation;
import javax.enterprise.inject.spi.Annotated;
import javax.enterprise.inject.spi.Bean;
import javax.enterprise.inject.spi.BeforeBeanDiscovery;
import javax.enterprise.inject.spi.BeforeShutdown;
import javax.enterprise.inject.spi.Extension;
import javax.enterprise.inject.spi.ProcessBean;

import li.rudin.core.scheduler.annotation.Schedule;

import org.quartz.CronScheduleBuilder;
import org.quartz.CronTrigger;
import org.quartz.JobBuilder;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.TriggerBuilder;
import org.quartz.impl.StdSchedulerFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScheduleExtension implements Extension
{
	private Scheduler scheduler;
	
	/**
	 * Local logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(ScheduleExtension.class);

	public void init(@Observes BeforeBeanDiscovery event)
	{
		try
		{
			scheduler = StdSchedulerFactory.getDefaultScheduler();
		}
		catch (SchedulerException e)
		{
			throw new IllegalArgumentException(e);
		}
	}
	
	private void scheduleCronJob(Schedule scheduleAnnotation, Bean<?> bean) throws SchedulerException
	{
		String cronString = 
				scheduleAnnotation.seconds() + " " +
				scheduleAnnotation.minutes() + " " +
				scheduleAnnotation.hours() + " " +
				scheduleAnnotation.daysOfMonth() + " " +
				scheduleAnnotation.months() + " " +
				scheduleAnnotation.daysOfWeek() + " " +
				scheduleAnnotation.years();
		
		JobDetail detail = JobBuilder
				.newJob(CDIJob.class)
				.usingJobData(CDIJob.JOB_MAP_CLASS_NAME, bean.getBeanClass().getName())
				.build();

		logger.info("Scheduling job '{}' on '{}'", bean.getBeanClass().getName(), cronString);
		
		CronTrigger trigger = TriggerBuilder
		.newTrigger()
		.withIdentity(bean.getBeanClass().getName())
		.withSchedule(CronScheduleBuilder.cronSchedule(cronString))
		.forJob(detail)
		.build();

		scheduler.scheduleJob(detail, trigger);
	}
	
	
	public <T> void collect(@Observes ProcessBean<T> event) {
		try
		{
			Annotated annotated = event.getAnnotated();

			Schedule scheduleAnnotation = getScheduleAnnotation(
					annotated.getAnnotations().toArray(
							new Annotation[annotated.getAnnotations().size()]
					),
					2
			);

			if (scheduleAnnotation != null && annotated.isAnnotationPresent(ApplicationScoped.class))
			{
				Bean<T> bean = event.getBean();
				scheduleCronJob(scheduleAnnotation, bean);
			}
			
		}
		catch (Exception e)
		{
			event.addDefinitionError(e);
		}
	}
	
	private Schedule getScheduleAnnotation(Annotation[] annotations, int levels)
	{
		if (levels == 0)
			return null;
		
		for (Annotation annotation: annotations)
		{
			Class<? extends Annotation> type = annotation.annotationType();
			if (type == Schedule.class)
				return (Schedule) annotation;
			
			Annotation[] subAnnotations = annotation.annotationType().getAnnotations();
			Schedule schedule = getScheduleAnnotation(subAnnotations, levels-1);
			if (schedule != null)
				return schedule;
		}
		
		return null;
	}

	public void start(@Observes AfterDeploymentValidation event)
	{
		try
		{
			scheduler.start();
		}
		catch (SchedulerException e)
		{
			event.addDeploymentProblem(e);
		}
	}

	public void stop(@Observes BeforeShutdown event)
	{
		try
		{
			scheduler.shutdown();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}
}