package li.rudin.core.security.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.enterprise.context.SessionScoped;
import javax.inject.Inject;

import li.rudin.core.security.AbstractUser;
import li.rudin.core.security.UserService;
import li.rudin.core.security.impl.schema.tables.Role;
import li.rudin.core.security.impl.schema.tables.User;
import li.rudin.core.security.impl.schema.tables.UserRoles;
import li.rudin.core.security.impl.schema.tables.records.RoleRecord;

import org.jooq.DSLContext;
import org.jooq.Record4;
import org.jooq.Result;

@SessionScoped
public class LoginServiceImpl implements UserService, Serializable
{
	/**
	 * Anonymous user
	 */
	public static final DefaultUser ANONYMOUS_USER = new DefaultUser("anonymous", "");

	private AbstractUser user = ANONYMOUS_USER;

	@Inject DSLContext ctx;

	@Override
	public AbstractUser getUser()
	{
		return user;
	}

	@Override
	public void login(AbstractUser user)
	{
		this.user = user;
	}

	@Override
	public void logout()
	{
		this.user = ANONYMOUS_USER;
	}

	@Override
	public boolean isLoggedIn()
	{
		return user != ANONYMOUS_USER;
	}

	@Override
	public List<AbstractUser> findByName(String name)
	{
		Result<Record4<Long, String,String,String>> records = ctx
				.select(User.USER.ID, User.USER.NAME, User.USER.SHA1, Role.ROLE.NAME)
				.from(User.USER.as("u"), Role.ROLE.as("r"), UserRoles.USER_ROLES)
				.join(User.USER)
				.on(User.USER.ID.eq(UserRoles.USER_ROLES.USER_ID))
				.join(Role.ROLE)
				.on(Role.ROLE.ID.eq(UserRoles.USER_ROLES.ROLE_ID))
				.fetch();

		List<AbstractUser> list = new ArrayList<>();

		DefaultUser user = null;
		Long currentId = null;

		for (Record4<Long, String,String,String> record: records)
		{
			Long id = record.getValue(User.USER.ID);

			if (currentId == null || currentId != id)
			{
				user = new DefaultUser();
				user.setName(record.getValue(User.USER.NAME));
				user.setSha1(record.getValue(User.USER.SHA1));
				list.add(user);
				
				currentId = id;
			}

			user.getRoles().add(record.getValue(Role.ROLE.NAME));
		}

		return list;
	}

	@Override
	public AbstractUser create(String name, String sha1, String... roles)
	{
		DefaultUser user = new DefaultUser(name, sha1, roles);

		Long id = ctx
				.insertInto(User.USER, User.USER.NAME, User.USER.SHA1)
				.values(name, sha1)
				.returning(User.USER.ID)
				.fetchOne()
				.getId();

		synchronized(Role.ROLE)
		{
			RoleRecord record;

			for (String role: roles)
			{
				record = ctx
						.selectFrom(Role.ROLE)
						.where(Role.ROLE.NAME.eq(role))
						.fetchOne();

				if (record == null)
				{
					record = ctx
							.insertInto(Role.ROLE, Role.ROLE.NAME)
							.values(role)
							.returning(Role.ROLE.ID)
							.fetchOne();
				}

				ctx
				.insertInto(UserRoles.USER_ROLES, UserRoles.USER_ROLES.USER_ID, UserRoles.USER_ROLES.ROLE_ID)
				.values(id, record.getId())
				.execute();
			}
		}

		return user;
	}
	
	@Override
	public void remove(AbstractUser user)
	{
		// TODO Auto-generated method stub

	}

	@Override
	public void update(AbstractUser user)
	{
		// TODO Auto-generated method stub

	}


}
