/**
 * RT Javascript client
 * for RT Servlet
 * Version 1.3-SNAPSHOT
 */
(function(global){

	/**
	 * Extender function
	 */
	var extend = function(base, defaults){
		for (var name in defaults){
			if (typeof(base[name]) === "undefined")
				//no default value in base, use from defaults
				base[name] = defaults[name];
		}
		//No recursion (yet)
	};
	
	/**
	 * JSONP requester
	 */
	var JSONP = function(opts){
		
		extend(opts, {
			url: "",
			timeout: 35000,
			success: function(data){},
			error: function(){}
		});
		
		//http://stackoverflow.com/questions/5907777/help-on-making-and-handling-jsonp-request-usin-javascript
		var cbName = "jsonp" + Math.floor( Math.random() * 1000000 ),
			script = document.createElement('SCRIPT'),
			//Cleanup job
			cleanup = function(){
				delete global[cbName];
				document.body.removeChild(script);
			},
			//Start timeout job
			timeoutJob = setTimeout(function(){
				cleanup();
				//error state
				opts.error();
			}, opts.timeout);
			
		//Source attribute with callback
		script.src = opts.url.replace("callback=?", "callback=" + cbName);
		
		//Global callback
		global[cbName] = function(data){
			
			//cancel timeout
			clearTimeout(timeoutJob);
			
			//callback
			opts.success(data);
			
			//cleanup
			cleanup();
		};
		
		//Start "request"
		document.body.appendChild(script);
	};

	/**
	 * RT with long poll implementation
	 */
	var RTLongPollImpl = function(url){
		var id = Math.floor( Math.random() * 10000 );

		//enhance url
		url = url + "?id=" + id + "&mode=longpoll&callback=?";
		
		var listeners = {};
		
		var doRequest = function(){
			JSONP({
				url: url,
				success: handler,
				error: doRequest //start request again after timeout
			});
		};
		
		var handler = function(list){
			if (list instanceof Array)
				list.forEach(function(e){
					var l = listeners[e.type];
					if (typeof l !== "undefined")
						l.forEach(function(l){
							l(e.data);
						});
				});
			doRequest();
		};
		
		//Start polling
		doRequest();
		
		this.on = function(name, fn){
			if (typeof listeners[name] === "undefined")
				listeners[name] = [];
			listeners[name].push(fn);
		};
	};

	/**
	 * RT with EventSource implementation
	 */
	var RTEventSourceImpl = function(url, error){
		var id = Math.floor( Math.random() * 10000 ),
			source = new EventSource(url + "?id=" + id + "&mode=sse");
		
		this.on = function(name, fn){
			source.addEventListener(name, function(e){
				fn(JSON.parse(e.data));
			});
		};
		
		setTimeout(function(){
			//Check connection, fail over to longpoll
			if (source.readyState === EventSource.OPEN)
				//ok
				return;
			
			//Close source
			source.close();
			error();			
		}, 2000);
		
	
	};
	
	/**
	 * Linked observable
	 * Lightweight version, without knockoutjs
	 */
	var Observable = function(rt, changeId){
		
		var listeners = [],
			value;
		
		/**
		 * Register observable
		 */
		rt.on(changeId, function(newValue){
			var oldValue = value;
			value = newValue;
			listeners.forEach(function(l){
				l(newValue, oldValue);
			});
		});
		
		//Expose read-only function
		var fn = function(){
			return value;
		};

		/**
		 * Subscribes a listener function
		 */
		fn.subscribe = function(listener){
			listeners.push(listener);
		};
		
		return fn;
	};
	

	/**
	 * RT facade
	 */
	var RTFacade = function(url){
		
		var self = this,
			defaultObservableFactory,
			impl,
			setImpl = function(i){
				impl = i;
				
				
				//copy over eventlisteners from previous implementation
				for (var name in listenerMap){
					listenerMap[name].forEach(function(l){
						i.on(name, l);
					});
				}
				//Reset map
				listenerMap = {};
			},
			listenerMap = {};
		
		this.on = function(name, listener){
		
			if (impl !== undefined)
				//Dispatch to implementation
				impl.on(name, listener);
			
			//Cache listener
			if (listenerMap[name] === undefined)
				listenerMap[name] = [];
			listenerMap[name].push(listener);
		};
		
		
		/**
		 * Sets the observable factory in this context
		 */
		this.setObservableFactory = function(factory){
			defaultObservableFactory = factory;
		};
			
		/**
		 * Generate linked observable
		 */
		this.observable = function(changeId, observableFactory){
			
			var factory = observableFactory;
			
			if (typeof(factory) === "undefined")
				//Set context wide factory
				factory = defaultObservableFactory;
				
			if (typeof(factory) === "undefined")
				//Return lightweight observable implementation
				return new Observable(self, changeId);
			else {
				//Use observable from specified factory (knockout-js)
				var o = factory();
				
				//Register observable listener
				self.on(changeId, function(e){
					//Set new value
					o(e);
				});
				
				return o;
			}
				
		};

		/**
		 * Generate linked observable arry
		 */
		this.observableArray = function(changeId){
			
			if (typeof(ko) === "undefined")
				throw "knockoutjs required for ObservableArray feature";
			
			var o = ko.observableArray();
				
			//Register observable listener
			self.on(changeId, function(e){
				
				switch(e.op){
				case "INIT":
					e.item.forEach(function(item){
						o.push(item);
					});
					break;
				case "APPEND":
					o.push(e.item);
					break;
				case "ADD":
					//TODO
					break;
				case "REMOVE":
					o.remove( o()[e.index] );
					break;
				case "REPLACE":
					//TODO
					break;
				case "CLEAR":
					o.removeAll();
					break;
				}
				
			});
				
			return o;
				
		};

		if (typeof EventSource === "undefined")
			//Use long polling
			setImpl(new RTLongPollImpl(url));
		else
			//Use event source
			setImpl(new RTEventSourceImpl(url, function(){
					//Fallback to long poll
					setImpl(new RTLongPollImpl(url));
				})
			);
		
	};
	

	//Export RT function
	if (typeof define === 'function' && define.amd)
		define(function() { return RTFacade; });
	else if (typeof module !== 'undefined' && module.exports)
		module.exports = RTFacade;
	else
		global.RT = RTFacade;

})(this);