package io.rudin.rt.api.observable.list;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

public class ObservableList<T> implements List<T>
{

	/**
	 * Creates an observer with backed ArrayList
	 */
	public ObservableList()
	{
		this(new ArrayList<T>());
	}
	
	/**
	 * Creates an observer with given default value
	 * @param list
	 */
	public ObservableList(List<T> list)
	{
		this.list = list;
	}
	
	/**
	 * All local listeners
	 */
	private Set<ListListener<T>> listListeners = new CopyOnWriteArraySet<>();
	
	/**
	 * The current value
	 */
	private final List<T> list;

	/**
	 * Returns the backed list
	 * @return
	 */
	public List<T> getList()
	{
		return list;
	}

	public void addListListener(ListListener<T> l)
	{
		listListeners.add(l);
	}

	public void removeListListener(ListListener<T> l)
	{
		listListeners.remove(l);
	}

	@Override
	public boolean add(T e)
	{
		for (ListListener<T> ll: listListeners)
			ll.onAdd(e);
		
		return list.add(e);
	}

	@Override
	public void add(int index, T element)
	{
		for (ListListener<T> ll: listListeners)
			ll.onAdd(element, index);
		list.add(index, element);
	}

	@Override
	public boolean addAll(Collection<? extends T> c) {
		
		for (ListListener<T> ll: listListeners)
			for (T obj: c)
				ll.onAdd(obj);
		
		return list.addAll(c);
	}

	@Override
	public boolean addAll(int index, Collection<? extends T> c) {
		throw new IllegalArgumentException("Sorry, not implemented, yet");
	}

	@Override
	public void clear() {
		for (ListListener<T> ll: listListeners)
			ll.onClear();
		
		list.clear();
	}

	@Override
	public boolean contains(Object o) {
		return list.contains(o);
	}

	@Override
	public boolean containsAll(Collection<?> c) {
		return list.containsAll(c);
	}

	@Override
	public T get(int index) {
		return list.get(index);
	}

	@Override
	public int indexOf(Object o) {
		return list.indexOf(o);
	}

	@Override
	public boolean isEmpty() {
		return list.isEmpty();
	}

	@Override
	public Iterator<T> iterator() {
		return list.iterator();
	}

	@Override
	public int lastIndexOf(Object o) {
		return list.lastIndexOf(o);
	}

	@Override
	public ListIterator<T> listIterator() {
		return list.listIterator();
	}

	@Override
	public ListIterator<T> listIterator(int index) {
		return list.listIterator(index);
	}

	@Override
	public boolean remove(Object o) {
		remove(list.indexOf(o));
		return true;
	}

	@Override
	public T remove(int index) {
		for (ListListener<T> ll: listListeners)
			ll.onRemove(index);
			
		return list.remove(index);
	}

	@Override
	public boolean removeAll(Collection<?> c) {
		for (Object o: c)
			remove(o);
		return true;
	}

	@Override
	public boolean retainAll(Collection<?> c) {
		
		for (T t: list)
			if (!c.contains(t))
				remove(t);
		
		return true;
	}

	@Override
	public T set(int index, T element) {
		for (ListListener<T> ll: listListeners)
			ll.onReplace(element, index);
		
		return list.set(index, element);
	}

	@Override
	public int size() {
		return list.size();
	}

	@Override
	public List<T> subList(int fromIndex, int toIndex) {
		throw new IllegalArgumentException("Sorry, not implemented, yet");
	}

	@Override
	public Object[] toArray() {
		return list.toArray();
	}

	@Override
	public <O> O[] toArray(O[] a) {
		return list.toArray(a);
	}

}
