package li.rudin.webdoc.core.template;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import li.rudin.webdoc.core.callback.TransformerCallback;
import li.rudin.webdoc.core.transformer.Transformer;
import li.rudin.webdoc.core.transformer.impl.HTMLTransformer;
import li.rudin.webdoc.core.transformer.impl.MarkdownTransformer;

/**
 * http://stackoverflow.com/questions/6214703/copy-entire-directory-contents-to-another-directory
 * @author user
 *
 */
public class TransformFileVisitor extends SimpleFileVisitor<Path>
{
	private final Path targetPath;
	private final TemplateTransformer transformer;
	private final TransformerCallback callback;
	
	private static final Transformer[] MARKUP_TRANSFORMERS = new Transformer[]{
		new MarkdownTransformer(),
		new HTMLTransformer()
	};
	
	private Path sourcePath = null;
	
	public TransformFileVisitor(Path targetPath, TemplateTransformer transformer, TransformerCallback callback)
	{
		this.targetPath = targetPath;
		this.callback = callback;
		this.transformer = transformer;
	}

	@Override
	public FileVisitResult preVisitDirectory(final Path dir, final BasicFileAttributes attrs) throws IOException
	{
		if (sourcePath == null)
			sourcePath = dir;
		else
			Files.createDirectories(targetPath.resolve(sourcePath.relativize(dir)));
		
		return FileVisitResult.CONTINUE;
	}

	@Override
	public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException
	{
		//Obtain basedir
		String baseDir = "";
		
		Path relativePath = sourcePath.relativize(file);
		for (int i=1; i<relativePath.getNameCount(); i++)
			baseDir += "../";
		
		Path newPath = targetPath.resolve(sourcePath.relativize(file));

		transformer.getVariables().put("basedir", baseDir);

		//Mapping found
		File sourceFile = file.toFile();
		String outputData = null;
		
		for (Transformer t: MARKUP_TRANSFORMERS)
		{
			if (file.toString().endsWith(t.getFileExtension() ))
			{

				//Transform with markup transformer
				outputData = t.transform(sourceFile);
				
				Path relativeFilePath = sourcePath.relativize(file);
				String newFileString = relativeFilePath.toString().substring(0, relativeFilePath.toString().length() - t.getFileExtension().length());
				newFileString += t.getTargetFileExtension();
				
				//Use extension
				newPath = targetPath.resolve(newFileString);
				
				if (t.wrapTemplate())
				{
					//Transform with template transformer
					outputData = transformer.transform(outputData);
				}
				
				break;
			}
		}
		
		if (outputData == null)
		{
			//No transformer found, just replace variables
			HTMLTransformer t = new HTMLTransformer();
			outputData = t.transform(sourceFile);
			
			outputData = transformer.replaceVariables(outputData);
		}
	
		callback.callback(file.toString(), newPath.toString());

		
		File newFile = newPath.toFile();
		newFile.getParentFile().mkdirs();
		FileOutputStream outputStream = new FileOutputStream(newFile);
		
		
		outputStream.write(outputData.getBytes());
		outputStream.close();

		return FileVisitResult.CONTINUE;
	}
}