package li.rudin.webdoc.plugin;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import li.rudin.webdoc.core.WebdocTransformer;
import li.rudin.webdoc.core.callback.TransformerCallback;
import li.rudin.webdoc.core.util.StreamUtil;

import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.model.Dependency;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

@Mojo(name="generate")
public class WebdocMojo extends AbstractMojo implements TransformerCallback
{

	//http://olamy.blogspot.fr/2012/05/java5-annotations-support-for-maven.html
	//https://cwiki.apache.org/confluence/display/MAVEN/Java+5+Annotations+for+Plugins
	//http://docs.codehaus.org/display/MAVENUSER/Mojo+Developer+Cookbook

	/**
	 * The template directory (must contain a template.html file)
	 */
	@Parameter(defaultValue="${project.build.directory}/../src/webdoc/template") String templateDir;
	
	/**
	 * The template artifact (optional): groupId:artifactId
	 */
	@Parameter String templateArtifact;
	
	/**
	 * The directory with the actual content
	 */
	@Parameter(defaultValue="${project.build.directory}/../src/webdoc/content") String contentDir;

	/**
	 * The output directory
	 */
	@Parameter(defaultValue="${project.build.directory}/webdoc") String outputDir;

	/**
	 * Variables
	 */
	@Parameter Map<String, String> variables;


	@Component MavenProject project;
	@Parameter(defaultValue="${localRepository}") ArtifactRepository localRepository;
	@Parameter(defaultValue="${project.dependencies}") List<Dependency> dependencies;

	/**
	 * Unzips the contents of the zip file to the target dir
	 * @param zipFile
	 * @param dir
	 * @throws IOException 
	 * @throws ZipException 
	 */
	private void unzipContents(File zipFile, String dir) throws ZipException, IOException
	{
		Log logger = getLog();
		
		File targetDir = new File(dir);
		
		ZipFile zip = new ZipFile(zipFile);
		Enumeration<? extends ZipEntry> entries = zip.entries();
		
		while (entries.hasMoreElements())
		{
			ZipEntry element = entries.nextElement();
			
			InputStream inputStream = zip.getInputStream(element);
			
			File targetFile = new File(targetDir, element.getName());
			
			if (element.isDirectory())
			{
				targetFile.mkdirs();
			}
			else if (element.getName().startsWith("META-INF"))
			{
				continue;
			}
			else
			{
				logger.info("Extracting to: " + targetFile.getAbsolutePath());
				
				FileOutputStream outputStream = new FileOutputStream(targetFile);
				
				StreamUtil.streamCopy(inputStream, outputStream);
				
				outputStream.close();
				inputStream.close();
			}
			
			
			
		}
		
		zip.close();
	}
	
	@Override
	public void execute() throws MojoExecutionException, MojoFailureException
	{

		Log logger = getLog();
		

		logger.info("Generating Webdoc: " + contentDir);

		try
		{
			if (templateArtifact != null)
			{
				//Resolve template artifact
				
				String[] parts = templateArtifact.split("[:]");
				if (parts.length != 2)
					throw new IllegalArgumentException("template artifact must contain a single :");
				
				String searchGroupId = parts[0];
				String searchArtifactId = parts[1];
								
				String jarUrl = null;
				
				//Template string specifies a dependency jar
				for (Dependency dep: dependencies)
				{
					if (dep.getArtifactId().equals(searchArtifactId) && dep.getGroupId().equals(searchGroupId))
					{
						//Artifact found
						jarUrl = 
								localRepository.getBasedir() + "/" + 
								searchGroupId.replace(".", "/") + "/" + 
								searchArtifactId + "/" +
								dep.getVersion() + "/" +
								searchArtifactId + "-" + dep.getVersion() + ".jar";
						
						logger.info("template-jar-url: " + jarUrl);
						
						break;
					}
				}
				
				if (jarUrl == null)
					throw new IllegalArgumentException("template artifact not found: " + templateArtifact);
				
				File zip = new File(jarUrl);
				
				if (!zip.exists())
					throw new FileNotFoundException("template jar not found: " + jarUrl);
				
				templateDir = project.getBuild().getDirectory() + "/webdoc-template";
				
				unzipContents(zip, templateDir);
				
				//TODO: dependencies
			}
			
			WebdocTransformer transformer = new WebdocTransformer(
					templateDir,
					contentDir
					);

			Map<String, String> vars = transformer.getVariables();

			if (variables != null)
				//put configured variables
				vars.putAll(variables);

			for (Map.Entry<Object, Object> entry: project.getProperties().entrySet())
				//Put property variables
				vars.put(entry.getKey().toString(), entry.getValue().toString());

			//Put build variables
			vars.put("project.version", project.getVersion());
			vars.put("project.name", project.getName());

			transformer.transform(outputDir, this);
		}
		catch (Exception e)
		{
			throw new MojoExecutionException("execute", e);
		}
	}

	@Override
	public void callback(String source, String target)
	{
		getLog().info(source + " -> " + target);
	}

}
