package li.rudin.webdoc.plugin;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import li.rudin.webdoc.core.WebdocTransformer;
import li.rudin.webdoc.core.callback.TransformerCallback;
import li.rudin.webdoc.core.util.StreamUtil;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Repository;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

@Mojo(name="generate")
public class GenerateMojo extends AbstractMojo implements TransformerCallback
{

	//http://olamy.blogspot.fr/2012/05/java5-annotations-support-for-maven.html
	//https://cwiki.apache.org/confluence/display/MAVEN/Java+5+Annotations+for+Plugins
	//http://docs.codehaus.org/display/MAVENUSER/Mojo+Developer+Cookbook

	/**
	 * The template directory (must contain a template.html file)
	 */
	@Parameter(defaultValue="${project.build.directory}/../src/webdoc/template") String templateDir;
	
	/**
	 * The template artifact (optional): groupId:artifactId
	 */
	@Parameter String templateArtifact;
	
	/**
	 * The directory with the actual content
	 */
	@Parameter(defaultValue="${project.build.directory}/../src/webdoc/content") String contentDir;

	/**
	 * The output directory
	 */
	@Parameter(defaultValue="${project.build.directory}/webdoc") String outputDir;

	/**
	 * Variables
	 */
	@Parameter Map<String, String> variables;


	@Component MavenProject project;
	@Parameter(defaultValue="${localRepository}") ArtifactRepository localRepository;
	@Parameter(defaultValue="${project.dependencies}") List<Dependency> dependencies;
	
	@Component ArtifactFactory artifactFactory;
	@Component ArtifactResolver resolver;
	@Component ArtifactMetadataSource artifactMetadataSource;

	/** @parameter default-value="${project.remoteArtifactRepositories}" */
	@Parameter(defaultValue="${project.remoteArtifactRepositories}") List<Repository> remoteRepositories;

	/**
	 * Unzips the contents of the zip file to the target dir
	 * @param zipFile
	 * @param dir
	 * @throws IOException 
	 * @throws ZipException 
	 */
	private void unzipContents(File zipFile, String dir) throws ZipException, IOException
	{
		Log logger = getLog();
		
		File targetDir = new File(dir);
		
		ZipFile zip = new ZipFile(zipFile);
		Enumeration<? extends ZipEntry> entries = zip.entries();
		
		while (entries.hasMoreElements())
		{
			ZipEntry element = entries.nextElement();
			
			InputStream inputStream = zip.getInputStream(element);
			
			File targetFile = new File(targetDir, element.getName());
			
			if (element.getName().startsWith("META-INF"))
				continue;
			else if (element.isDirectory())
				targetFile.mkdirs();
			else
			{
				logger.info("Extracting to: " + targetFile.getAbsolutePath());
				
				FileOutputStream outputStream = new FileOutputStream(targetFile);
				
				StreamUtil.streamCopy(inputStream, outputStream);
				
				outputStream.close();
				inputStream.close();
			}
			
			
			
		}
		
		zip.close();
	}
	
	@Override
	public void execute() throws MojoExecutionException, MojoFailureException
	{

		Log logger = getLog();
		

		logger.info("Generating Webdoc: " + contentDir);

		try
		{
			if (templateArtifact != null)
			{
				//Resolve template artifact
				
				String[] parts = templateArtifact.split("[:]");
				if (parts.length != 3)
					throw new IllegalArgumentException("template artifact must be in form: <groupId>:<artifactId>:<version>");
				
				String groupId = parts[0];
				String artifactId = parts[1];
				String version = parts[2];
									
				Artifact artifact = artifactFactory.createArtifact(groupId, artifactId, version, "compile", "jar");
				
				resolver.resolve( artifact, remoteRepositories, localRepository );
				
				File artifactFile = artifact.getFile();
				
				logger.info("Template artifact: " + artifactFile.getAbsolutePath());
				
				templateDir = project.getBuild().getDirectory() + "/webdoc-template";
				
				unzipContents(artifactFile, templateDir);
				
				//TODO: dependencies
			}
			
			WebdocTransformer transformer = new WebdocTransformer(
					templateDir,
					contentDir
					);

			Map<String, String> vars = transformer.getVariables();

			if (variables != null)
				//put configured variables
				vars.putAll(variables);

			for (Map.Entry<Object, Object> entry: project.getProperties().entrySet())
				//Put property variables
				vars.put(entry.getKey().toString(), entry.getValue().toString());

			//Put build variables
			vars.put("project.version", project.getVersion());
			vars.put("project.name", project.getName());

			transformer.transform(outputDir, this);
		}
		catch (Exception e)
		{
			logger.error(e);
			throw new MojoExecutionException("execute", e);
		}
	}

	@Override
	public void callback(String source, String target)
	{
		getLog().info(source + " -> " + target);
	}

}
