package li.rudin.webdoc.plugin;

import static org.twdata.maven.mojoexecutor.MojoExecutor.artifactId;
import static org.twdata.maven.mojoexecutor.MojoExecutor.configuration;
import static org.twdata.maven.mojoexecutor.MojoExecutor.element;
import static org.twdata.maven.mojoexecutor.MojoExecutor.executeMojo;
import static org.twdata.maven.mojoexecutor.MojoExecutor.executionEnvironment;
import static org.twdata.maven.mojoexecutor.MojoExecutor.goal;
import static org.twdata.maven.mojoexecutor.MojoExecutor.groupId;
import static org.twdata.maven.mojoexecutor.MojoExecutor.name;
import static org.twdata.maven.mojoexecutor.MojoExecutor.plugin;
import static org.twdata.maven.mojoexecutor.MojoExecutor.version;

import java.util.List;
import java.util.Map;

import li.rudin.webdoc.core.WebdocTransformer;
import li.rudin.webdoc.core.callback.TransformerCallback;

import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Repository;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

@Mojo(name="generate")
public class GenerateMojo extends AbstractMojo implements TransformerCallback
{

	//http://olamy.blogspot.fr/2012/05/java5-annotations-support-for-maven.html
	//https://cwiki.apache.org/confluence/display/MAVEN/Java+5+Annotations+for+Plugins
	//http://docs.codehaus.org/display/MAVENUSER/Mojo+Developer+Cookbook

	/**
	 * The template directory (must contain a template.html file)
	 */
	@Parameter(defaultValue="${project.build.directory}/../src/webdoc/template") String templateDir;

	/**
	 * The template artifact (optional): groupId:artifactId
	 */
	@Parameter String templateArtifact;

	/**
	 * The directory with the actual content
	 */
	@Parameter(defaultValue="${project.build.directory}/../src/webdoc/content") String contentDir;

	/**
	 * The output directory
	 */
	@Parameter(defaultValue="${project.build.directory}/webdoc") String outputDir;

	/**
	 * Variables
	 */
	@Parameter Map<String, String> variables;


	@Component MavenProject project;
	@Parameter(defaultValue="${localRepository}") ArtifactRepository localRepository;
	@Parameter(defaultValue="${project.dependencies}") List<Dependency> dependencies;


	/** @parameter default-value="${project.remoteArtifactRepositories}" */
	@Parameter(defaultValue="${project.remoteArtifactRepositories}") List<Repository> remoteRepositories;

	@Parameter(defaultValue="${session}") MavenSession session;
	@Component BuildPluginManager pluginManager;


	@Override
	public void execute() throws MojoExecutionException, MojoFailureException
	{

		Log logger = getLog();


		logger.info("Generating Webdoc: " + contentDir);

		try
		{
			if (templateArtifact != null)
			{
				//Resolve template artifact

				String[] parts = templateArtifact.split("[:]");
				if (parts.length != 3)
					throw new IllegalArgumentException("template artifact must be in form: <groupId>:<artifactId>:<version>");

				String groupId = parts[0];
				String artifactId = parts[1];
				String version = parts[2];


				//Override template dir
				templateDir = project.getBuild().getDirectory() + "/webdoc-template";


				executeMojo(
						plugin(
								groupId("org.apache.maven.plugins"),
								artifactId("maven-dependency-plugin"),
								version("2.8")
								),
								goal("unpack"),
								configuration(
										element(name("outputDirectory"), templateDir),
										element(name("artifactItems"),
												element(name("artifactItem"), 
														element("groupId", groupId),
														element("artifactId", artifactId),
														element("version", version)
														)
												)
										),
										executionEnvironment(
												project,
												session,
												pluginManager
												)
						);

			}

			WebdocTransformer transformer = new WebdocTransformer(
					templateDir,
					contentDir
					);

			Map<String, String> vars = transformer.getVariables();

			if (variables != null)
				//put configured variables
				vars.putAll(variables);

			for (Map.Entry<Object, Object> entry: project.getProperties().entrySet())
				//Put property variables
				vars.put(entry.getKey().toString(), entry.getValue().toString());

			//Put build variables
			vars.put("project.version", project.getVersion());
			vars.put("project.name", project.getName());

			transformer.transform(outputDir, this);
		}
		catch (Exception e)
		{
			logger.error(e);
			throw new MojoExecutionException("execute", e);
		}
	}

	@Override
	public void callback(String source, String target)
	{
		getLog().info(source + " -> " + target);
	}

}
