package li.rudin.webdoc.plugin;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.List;

import li.rudin.webdoc.plugin.visitor.FTPUploadFileVisitor;

import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPReply;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;

@Mojo(name="upload")
public class UploadMojo extends AbstractMojo
{

	//http://stackoverflow.com/questions/6527664/how-do-you-upload-a-file-to-an-ftp-server

	/**
	 * The files/folders to upload
	 */
	@Parameter(required=true) List<File> files;

	/**
	 * The remote host to use
	 */
	@Parameter(required=true) String remoteHost;

	@Parameter(defaultValue="") String remoteDir;
	@Parameter(defaultValue="21") int remotePort;

	/**
	 * Username (optional)
	 */
	@Parameter String username;

	/**
	 * Password (optional)
	 */
	@Parameter String password;

	/**
	 * The server id for the credentials
	 */
	@Parameter String serverId;

	@Component Settings settings;

	private FTPClient client;

	@Override
	public void execute() throws MojoExecutionException, MojoFailureException
	{

		Log logger = getLog();

		if (files == null || files.size() == 0)
			//Nothing to upload
			return;

		try
		{
			client = new FTPClient();
			client.connect(remoteHost, remotePort);

			boolean connected = false;

			if (serverId != null)
			{
				Server server = settings.getServer(serverId);
				if (server == null)
					logger.error("server config not found: " + serverId);
				
				connected = client.login(server.getUsername(), server.getPassword());
			}
			else if (username != null && password == null)
			{
				connected = client.login(username, "");
			}
			else if (username != null && password != null)
			{
				connected = client.login(username, password);
			}

			if (!connected)
				throw new IllegalArgumentException("not connected!");

			logger.info("Connected to: " + remoteHost + ":" + remotePort);

			client.setFileType(FTP.BINARY_FILE_TYPE);
			client.enterLocalPassiveMode();

			if (remoteDir.length() > 0 && !remoteDir.endsWith("/"))
				//Path must end with a slash
				remoteDir += "/";


			for (File file: files)
			{
				if (file.isDirectory())
					Files.walkFileTree(file.toPath(), new FTPUploadFileVisitor(this));
				else if (file.isFile())
				{
					FileInputStream inputStream = new FileInputStream(file);
					upload(file.getName(), inputStream);
					inputStream.close();
				}
			}

			client.disconnect();
		}
		catch (Exception e)
		{
			logger.error(e);
			throw new IllegalArgumentException("upload", e);
		}
	}

	private void changeDir(String dir) throws IOException
	{
		if (dir.equals(currentDir))
			//Already there
			return;

		Log logger = getLog();
		logger.info("Changing dir: " + dir);

		if (currentDir != null)
		{
			//Go back to root
			String[] dirParts = currentDir.split("/");

			for (int i=0; i<dirParts.length; i++)
				if(!client.changeToParentDirectory())
					throw new IllegalArgumentException("could not change to parent dir");
		}

		for (String part: dir.split("/"))
		{
			if(!client.changeWorkingDirectory(part))
			{
				int result = client.mkd(part);
				if (result != FTPReply.PATHNAME_CREATED)
					throw new IllegalArgumentException("could not create path: " + part);

				if (!client.changeWorkingDirectory(part))
					throw new IllegalArgumentException("could not chnage to dir: " + part);
			}
		}

		currentDir = dir;
	}

	private String currentDir;

	public void upload(String targetName, InputStream file) throws IOException
	{
		Log logger = getLog();

		String fullName = remoteDir + targetName;
		String fileName = targetName;

		if (fullName.contains("/"))
		{
			//Subdirectories present
			int lastSlashIndex = fullName.lastIndexOf("/");
			String dir = fullName.substring(0, lastSlashIndex);
			changeDir(dir);
			fileName = fullName.substring(lastSlashIndex+1, fullName.length());
		}


		logger.info("Uploading: " + fullName + " ("+fileName+")");

		client.storeFile(fileName, file);
	}


}
