package me.adkhambek.gsa.compiler.generator

import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.KModifier
import com.squareup.kotlinpoet.PropertySpec
import com.squareup.kotlinpoet.TypeSpec
import me.adkhambek.gsa.compiler.model.ArgumentInfo
import me.adkhambek.gsa.compiler.model.ScreenInfo
import me.adkhambek.gsa.compiler.utils.ARG_BUNDLE
import me.adkhambek.gsa.compiler.utils.BUNDLE_CLASSNAME
import me.adkhambek.gsa.compiler.utils.NameUtil
import me.adkhambek.gsa.compiler.utils.toCamelCase
import me.adkhambek.gsa.compiler.utils.toCamelCaseAsVar
import me.adkhambek.gsa.compiler.writer.addBundleGetStatement
import me.adkhambek.gsa.compiler.writer.addBundlePutStatement

class ArgumentHolderGenerator {

    fun gen(info: List<ScreenInfo>): List<FileSpec> {
        return info
            .filter { it.arguments.isNotEmpty() }
            .map { screen ->
                genArguments(screen.fragment, screen.arguments)
            }
    }

    private fun genArguments(fragment: ClassName, arguments: List<ArgumentInfo>): FileSpec {
        val argumentClass = NameUtil.argumentClass(fragment)

        val companionBuilder = TypeSpec
            .companionObjectBuilder()
            .addFunction(genReadFunction(fragment, arguments))
            .genReadSingleArgFunction(arguments = arguments)
            .build()

        val classArg = TypeSpec
            .classBuilder(argumentClass)
            .addModifiers(KModifier.DATA)
            .addFunction(genWriteFunction(arguments))
            .addProperties(generateProperties(arguments))
            .primaryConstructor(generateConstructor(arguments))
            .addType(companionBuilder)
            .build()

        return FileSpec
            .builder(fragment.packageName, argumentClass)
            .addType(classArg)
            .build()
    }


    private fun generateConstructor(args: List<ArgumentInfo>): FunSpec {
        val constructor = FunSpec.constructorBuilder()

        args.forEach { arg ->
            constructor.addParameter(arg.name.toCamelCaseAsVar(), arg.type.copy(arg.isNullable))
        }

        return constructor.build()
    }

    private fun generateProperties(args: List<ArgumentInfo>): List<PropertySpec> {
        return args.map { arg ->
            PropertySpec.builder(arg.name.toCamelCaseAsVar(), arg.type.copy(arg.isNullable))
                .initializer(arg.name.toCamelCaseAsVar())
                .build()
        }
    }

    private fun genWriteFunction(arguments: List<ArgumentInfo>): FunSpec {
        val func = FunSpec
            .builder("toBundle")
            .returns(BUNDLE_CLASSNAME)
            .addStatement("val %L = %T(%L)", ARG_BUNDLE, BUNDLE_CLASSNAME, arguments.size)

        if (arguments.isNotEmpty()) {
            arguments.forEach { arg ->
                val argumentName = arg.name.toCamelCaseAsVar()
                arg.type.addBundlePutStatement(func, arg, ARG_BUNDLE, argumentName)
            }
        }

        func.addStatement("return %L", ARG_BUNDLE)
        return func.build()
    }

    private fun genReadFunction(fragment: ClassName, arguments: List<ArgumentInfo>): FunSpec {
        val args = "args"

        val argumentClass = NameUtil.argumentClass(fragment)
        val argumentClassName = ClassName(fragment.packageName, argumentClass)

        val func = FunSpec
            .builder("fromBundle")
            .addAnnotation(JvmStatic::class)
            .addParameter(ARG_BUNDLE, BUNDLE_CLASSNAME)
            .returns(argumentClassName)
            .addStatement("")

        if (arguments.isNotEmpty()) {
            arguments.forEach { arg ->
                val argumentName = arg.name.toCamelCaseAsVar()
                arg.type.addBundleGetStatement(func, arg, ARG_BUNDLE, argumentName)
            }
        }

        func
            .addCreateArgs(args, argumentClassName, arguments)
            .addStatement("return %L", args)

        return func.build()
    }

    private fun TypeSpec.Builder.genReadSingleArgFunction(arguments: List<ArgumentInfo>): TypeSpec.Builder {

        if (arguments.isNotEmpty()) {
            arguments.forEach { arg ->
                val func = FunSpec
                    .builder("get" + arg.name.toCamelCase())
                    .addAnnotation(JvmStatic::class)
                    .addParameter(ARG_BUNDLE, BUNDLE_CLASSNAME)
                    .returns(arg.type.copy(nullable = arg.isNullable))
                    .addStatement("")

                val argumentName = arg.name.toCamelCaseAsVar()
                arg.type.addBundleGetStatement(func, arg, ARG_BUNDLE, argumentName)

                func.addStatement("return %L", argumentName)
                addFunction(func.build())
            }
        }

        return this
    }
}

fun FunSpec.Builder.addCreateArgs(
    variable: String,
    argumentClassName: ClassName,
    arguments: List<ArgumentInfo>,
): FunSpec.Builder {

    addStatement("val %L = %T(", variable, argumentClassName)

    arguments.forEach { arg ->
        addStatement("\t%L = %L,", arg.name.toCamelCaseAsVar(), arg.name.toCamelCaseAsVar())
    }

    addStatement(")")
    return this
}

fun FunSpec.Builder.addArguments(
    arguments: List<ArgumentInfo>,
): FunSpec.Builder {

    arguments.forEach { arg ->
        addParameter(arg.name.toCamelCaseAsVar(), arg.type.copy(nullable = arg.isNullable))
    }

    return this
}