package me.adkhambek.gsa.compiler.generator

import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.TypeSpec
import me.adkhambek.gsa.compiler.model.ScreenInfo
import me.adkhambek.gsa.compiler.utils.ARG_BUNDLE
import me.adkhambek.gsa.compiler.utils.NameUtil
import me.adkhambek.gsa.compiler.utils.SCREEN_FRAGMENT_CLASSNAME
import me.adkhambek.gsa.compiler.utils.STRING_BUILDER_CLASSNAME

class ScreenGenerator {

    fun gen(info: List<ScreenInfo>): List<FileSpec> {
        val owners = info.groupBy(ScreenInfo::installIn)

        val screens: List<FileSpec> = owners.map { (installIn, data) ->
            val functions = data.map(::genFunction)
            val className = NameUtil.screensClass(installIn)

            val screenObject = TypeSpec
                .objectBuilder(className)
                .addFunctions(functions)
                .build()

            FileSpec
                .builder(installIn.packageName, className)
                .addType(screenObject)
                .build()
        }

        return screens
    }

    private fun genFunction(screenInfo: ScreenInfo): FunSpec {
        val fr = "fragment"
        val args = "args"
        val keyBuilder = "keyBuilder"

        val argumentClass = NameUtil.argumentClass(screenInfo.fragment)
        val argumentClassName = ClassName(screenInfo.fragment.packageName, argumentClass)

        val func = FunSpec
            .builder(NameUtil.screenClass(screenInfo.fragment))
            .addAnnotation(JvmStatic::class)
            .returns(SCREEN_FRAGMENT_CLASSNAME)
            .addStatement("")
            .addStatement("val %L = %T()", fr, screenInfo.fragment)
            .addStatement("val %L = %T(%S)", keyBuilder, STRING_BUILDER_CLASSNAME, screenInfo.fragment.canonicalName)

        if (screenInfo.arguments.isNotEmpty()) {
            func
                .addStatement("")
                .addArguments(screenInfo.arguments)
                .addCreateArgs(args, argumentClassName, screenInfo.arguments)
                .addStatement("val %L = %L.toBundle()", ARG_BUNDLE, args)
                .addStatement("%L.setArguments(%L)", fr, ARG_BUNDLE)
        }

        func
            .addStatement("")
            .addStatement(
                """
                return FragmentScreen(
                    clearContainer = %L,
                    key = %L.toString()
                ) { 
                    %L
                }
            """.trimIndent(), screenInfo.clearContainer, keyBuilder, fr
            )

        return func.build()
    }
}