package me.adkhambek.gsa

import com.google.auto.service.AutoService
import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.processing.SymbolProcessor
import com.google.devtools.ksp.processing.SymbolProcessorEnvironment
import com.google.devtools.ksp.processing.SymbolProcessorProvider
import com.google.devtools.ksp.symbol.KSAnnotated
import com.google.devtools.ksp.symbol.KSClassDeclaration
import me.adkhambek.gsa.compiler.generator.ArgumentHolderGenerator
import me.adkhambek.gsa.compiler.generator.ScreenGenerator
import me.adkhambek.gsa.compiler.writer.Writer
import me.adkhambek.gsa.parser.ArgumentParser
import me.adkhambek.gsa.parser.ParserVisitor
import me.adkhambek.gsa.parser.ScreenParser
import me.adkhambek.gsa.utils.SCREEN_TYPE_QUALIFIED_NAME
import me.adkhambek.gsa.validator.SymbolValidator

internal class CiceroneKspProcessor(
    private val processingEnv: SymbolProcessorEnvironment,
    private val validator: SymbolValidator,
) : SymbolProcessor {


    override fun process(resolver: Resolver): List<KSAnnotated> {
        processingEnv.logger.info("Start")

        val writer = Writer(
            screenGenerator = ScreenGenerator(),
            generator = processingEnv.codeGenerator,
            argumentHolderGenerator = ArgumentHolderGenerator(
                resolver = resolver
            )
        )

        val visitor = ParserVisitor(
            resolver = resolver,
            argumentVisitor = ScreenParser(
                argumentParser = ArgumentParser(resolver)
            )
        )

        val resolved = resolver.getSymbolsWithAnnotation(SCREEN_TYPE_QUALIFIED_NAME)

        resolved.forEach {
            processingEnv.logger.info(it.toString())
        }

        val validatedSymbols: List<KSClassDeclaration> = resolved
            .filterIsInstance<KSClassDeclaration>()
            .onEach {
                processingEnv.logger.info("Valid $it")
            }
            .filter {
                val t = validator.isValid(it)
                processingEnv.logger.info(t.toString())
                t
            }
            .toList()

        validatedSymbols
            .map { it.accept(visitor, Unit) }
            .let(writer::write)

        processingEnv.logger.info("Valid" + validatedSymbols.size)

        return resolved.toList() - validatedSymbols.toSet()
    }

    @AutoService(SymbolProcessorProvider::class)
    class Provider : SymbolProcessorProvider {
        override fun create(environment: SymbolProcessorEnvironment): SymbolProcessor {
            return CiceroneKspProcessor(
                processingEnv = environment,
                validator = SymbolValidator()
            )
        }
    }
}