package me.adkhambek.gsa.compiler.generator

import com.google.devtools.ksp.processing.Resolver
import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.KModifier
import com.squareup.kotlinpoet.ParameterizedTypeName.Companion.parameterizedBy
import com.squareup.kotlinpoet.PropertySpec
import com.squareup.kotlinpoet.TypeSpec
import com.squareup.kotlinpoet.asClassName
import com.squareup.kotlinpoet.ksp.toClassName
import me.adkhambek.gsa.compiler.writer.addBundleGetStatement
import me.adkhambek.gsa.compiler.writer.addBundlePutStatement
import me.adkhambek.gsa.core.Arguments
import me.adkhambek.gsa.core.ArgumentsParser
import me.adkhambek.gsa.model.ArgumentInfo
import me.adkhambek.gsa.model.ScreenInfo
import me.adkhambek.gsa.utils.ARG_BUNDLE
import me.adkhambek.gsa.utils.BUNDLE_CLASSNAME
import me.adkhambek.gsa.utils.NameUtil
import me.adkhambek.gsa.utils.addCreateArgs
import me.adkhambek.gsa.utils.toCamelCaseAsVar

class ArgumentHolderGenerator(
    private val resolver: Resolver,
) {

    fun gen(info: List<ScreenInfo>): List<FileSpec> {
        return info
            .filter { it.arguments.isNotEmpty() }
            .map { screen ->
                genArguments(screen.owner.toClassName(), screen.arguments)
            }
    }

    private fun genArguments(fragment: ClassName, arguments: List<ArgumentInfo>): FileSpec {
        val argumentClass = NameUtil.argumentClass(fragment)

        val companionBuilder = TypeSpec
            .companionObjectBuilder("Parser")
            .addSuperinterface(ArgumentsParser::class.parameterizedBy(Arguments::class))
            .addFunction(genReadFunction(fragment, arguments))
            .build()

        val classArg = TypeSpec
            .classBuilder(argumentClass)
            .addModifiers(KModifier.DATA)

            .addProperties(generateProperties(arguments))
            .primaryConstructor(generateConstructor(arguments))

            .addSuperinterface(Arguments::class.asClassName())
            .addFunction(genToBundleFunction(arguments))

            .addType(companionBuilder)
            .build()

        return FileSpec
            .builder(fragment.packageName, argumentClass)
            .addType(classArg)
            .build()
    }


    private fun generateConstructor(args: List<ArgumentInfo>): FunSpec {
        val constructor = FunSpec.constructorBuilder()

        args.forEach { arg ->
            val argClassName = arg.type.toClassName()
            constructor.addParameter(arg.name.toCamelCaseAsVar(), argClassName.copy(arg.isNullable))
        }

        return constructor.build()
    }

    private fun generateProperties(args: List<ArgumentInfo>): List<PropertySpec> {
        return args.map { arg ->
            val propertyClassName = arg.type.toClassName()
            PropertySpec.builder(arg.name.toCamelCaseAsVar(), propertyClassName.copy(arg.isNullable))
                .initializer(arg.name.toCamelCaseAsVar())
                .build()
        }
    }

    private fun genToBundleFunction(arguments: List<ArgumentInfo>): FunSpec {
        val func = FunSpec
            .builder("toBundle")
            .addModifiers(KModifier.OVERRIDE)
            .returns(BUNDLE_CLASSNAME)
            .addStatement("val %L = %T(%L)", ARG_BUNDLE, BUNDLE_CLASSNAME, arguments.size)

        if (arguments.isNotEmpty()) {
            arguments.forEach { arg ->
                arg.type.addBundlePutStatement(
                    resolver = resolver,
                    builder = func,
                    arg = arg,
                    bundle = ARG_BUNDLE
                )
            }
        }

        func.addStatement("return %L", ARG_BUNDLE)
        return func.build()
    }

    private fun genReadFunction(fragment: ClassName, arguments: List<ArgumentInfo>): FunSpec {
        val args = "args"

        val argumentClass = NameUtil.argumentClass(fragment)
        val argumentClassName = ClassName(fragment.packageName, argumentClass)

        val func = FunSpec
            .builder("fromBundle")
            .addModifiers(KModifier.OVERRIDE)
            .addAnnotation(JvmStatic::class)
            .addParameter(ARG_BUNDLE, BUNDLE_CLASSNAME)
            .returns(argumentClassName)
            .addStatement("")

        if (arguments.isNotEmpty()) {
            arguments.forEach { arg ->
                arg.type.addBundleGetStatement(
                    resolver = resolver,
                    builder = func,
                    arg = arg,
                    bundle = ARG_BUNDLE
                )
            }
        }

        func
            .addCreateArgs(args, argumentClassName, arguments)
            .addStatement("return %L", args)

        return func.build()
    }



}