package me.adkhambek.gsa.compiler.generator

import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FileSpec
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.TypeSpec
import com.squareup.kotlinpoet.ksp.toClassName
import me.adkhambek.gsa.model.ScreenInfo
import me.adkhambek.gsa.utils.ARG_BUNDLE
import me.adkhambek.gsa.utils.NameUtil
import me.adkhambek.gsa.utils.SCREEN_FRAGMENT_CLASSNAME
import me.adkhambek.gsa.utils.STRING_BUILDER_CLASSNAME
import me.adkhambek.gsa.utils.addArguments
import me.adkhambek.gsa.utils.addCreateArgs

class ScreenGenerator {

    fun gen(info: List<ScreenInfo>): List<FileSpec> {
        val owners = info.groupBy(ScreenInfo::installIn)

        val screens: List<FileSpec> = owners.map { (installIn, data) ->

            val installInClassName: ClassName = installIn.toClassName()
            val className = NameUtil.screensClass(installInClassName)
            val functions = data.map(::genFunction)

            val screenObject = TypeSpec
                .objectBuilder(className)
                .addFunctions(functions)
                .build()

            FileSpec
                .builder(installInClassName.packageName, className)
                .addType(screenObject)
                .build()
        }

        return screens
    }

    private fun genFunction(screenInfo: ScreenInfo): FunSpec {
        val keyBuilder = "keyBuilder"
        val fr = "fragment"
        val args = "args"

        val ownerClassName = screenInfo.owner.toClassName()
        val argumentClass = NameUtil.argumentClass(ownerClassName)
        val argumentClassName = ClassName(ownerClassName.packageName, argumentClass)

        val func = FunSpec
            .builder(NameUtil.screenClass(ownerClassName))
            .addAnnotation(JvmStatic::class)
            .returns(SCREEN_FRAGMENT_CLASSNAME)
            .addStatement("")
            .addStatement("val %L = %T(%S)", keyBuilder, STRING_BUILDER_CLASSNAME, ownerClassName.canonicalName)

        if (screenInfo.arguments.isNotEmpty()) {
            func
                .addStatement("")
                .addArguments(screenInfo.arguments)
                .addCreateArgs(args, argumentClassName, screenInfo.arguments)
                .addStatement("val %L = %L.toBundle()", ARG_BUNDLE, args)
        }

        func
            .addStatement("")
            .addStatement("""
                return FragmentScreen(
                    clearContainer = %L,
                    key = %L.toString()
                ) {
                    val %L = %T()
                    %L.setArguments(%L)
                    %L
                }
            """.trimIndent(),
                screenInfo.clearContainer,
                keyBuilder,
                fr, screenInfo.owner.toClassName(),
                fr, ARG_BUNDLE,
                fr
            )

        return func.build()
    }
}