package me.adkhambek.gsa.compiler.writer

import com.google.devtools.ksp.processing.Resolver
import com.google.devtools.ksp.symbol.KSType
import com.squareup.kotlinpoet.BOOLEAN
import com.squareup.kotlinpoet.BOOLEAN_ARRAY
import com.squareup.kotlinpoet.FLOAT
import com.squareup.kotlinpoet.FLOAT_ARRAY
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.INT
import com.squareup.kotlinpoet.INT_ARRAY
import com.squareup.kotlinpoet.LONG
import com.squareup.kotlinpoet.LONG_ARRAY
import com.squareup.kotlinpoet.STRING
import com.squareup.kotlinpoet.ksp.toClassName
import me.adkhambek.gsa.model.ArgumentInfo
import me.adkhambek.gsa.utils.isParcelable
import me.adkhambek.gsa.utils.isSerializable
import me.adkhambek.gsa.utils.toCamelCaseAsVar

internal fun KSType.addBundlePutStatement(
    resolver: Resolver,
    builder: FunSpec.Builder,
    arg: ArgumentInfo,
    bundle: String,
): FunSpec.Builder {

    val method = bundleBuiltInPutMethod()
        ?: bundleSupportPutMethod(resolver)
        ?: error("Type must be Parcelable/Serializable/Primitive")

    builder.addStatement(
        format = "%L.%L(%S, %L)",
        bundle,
        method,
        arg.name,
        arg.name.toCamelCaseAsVar()
    )

    return builder
}

internal fun KSType.bundleBuiltInPutMethod(): String? = when (this.toClassName()) {
    BOOLEAN -> "putBoolean"
    BOOLEAN_ARRAY -> "putBooleanArray"

    LONG -> "putLong"
    LONG_ARRAY -> "putLongArray"

    INT -> "putInt"
    INT_ARRAY -> "putIntArray"

    FLOAT -> "putFloat"
    FLOAT_ARRAY -> "putFloatArray"

    STRING -> "putString"
    else -> null
}

internal fun KSType.bundleSupportPutMethod(resolver: Resolver): String? = when {
    isParcelable(resolver = resolver) -> "putParcelable"
    isSerializable(resolver = resolver) -> "putSerializable"
    else -> null
}

////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////

const val NULLABLE_VAR: String = "val %L: %T? = %L.%L(%S) //  { %S }"
const val NON_NULLABLE_VAR: String = "val %L: %T = requireNotNull(%L.%L(%S)) { %S }"

internal fun KSType.addBundleGetStatement(
    resolver: Resolver,
    builder: FunSpec.Builder,
    arg: ArgumentInfo,
    bundle: String,
): FunSpec.Builder {

    val method: String = bundleBuiltInGetMethod()
        ?: bundleSupportGetMethod(resolver)
        ?: error("Type must be Parcelable/Serializable/Primitive")

    builder.addStatement(
        format = if (arg.isNullable) NULLABLE_VAR else NON_NULLABLE_VAR,
        arg.name.toCamelCaseAsVar(),
        this.toClassName(),
        bundle,
        method,
        arg.name,
        "Argument ${arg.name} not found"
    )

    return builder
}

internal fun KSType.bundleBuiltInGetMethod(): String? = when (this.toClassName()) {
    BOOLEAN -> "getBoolean"
    BOOLEAN_ARRAY -> "getBooleanArray"

    LONG -> "getLong"
    LONG_ARRAY -> "getLongArray"

    INT -> "getInt"
    INT_ARRAY -> "getIntArray"

    FLOAT -> "getFloat"
    FLOAT_ARRAY -> "getFloatArray"

    STRING -> "getString"
    else -> null
}

internal fun KSType.bundleSupportGetMethod(resolver: Resolver): String? = when {
    isParcelable(resolver = resolver) -> "getParcelable<${this.toClassName().simpleName}>"
    isSerializable(resolver = resolver) -> "getSerializable"
    else -> null
}