/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.cosid.shardingsphere.sharding.interval;

import me.ahoo.cosid.sharding.IntervalStep;
import me.ahoo.cosid.sharding.IntervalTimeline;
import me.ahoo.cosid.sharding.Sharding;
import me.ahoo.cosid.shardingsphere.sharding.CosIdAlgorithm;
import me.ahoo.cosid.shardingsphere.sharding.utils.PropertiesUtil;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.BoundType;
import com.google.common.collect.Range;
import org.apache.shardingsphere.sharding.api.sharding.standard.PreciseShardingValue;
import org.apache.shardingsphere.sharding.api.sharding.standard.RangeShardingValue;
import org.apache.shardingsphere.sharding.api.sharding.standard.StandardShardingAlgorithm;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Properties;

/**
 * 基于间隔的时间范围分片算法.
 * <pre>
 * 1. 易用性 支持多种数据类型 (Long:时间戳、LocalDateTime、DATE)，而官方的实现是先转换成字符串再转换成 LocalDateTime，转换成功率受时间格式化字符影响
 * 2. 性能
 * -- 2.1 算法复杂度:[O(1)]，而官方的实现使用的是遍历查找[O(N)] {org.apache.shardingsphere.sharding.algorithm.sharding.datetime.IntervalShardingAlgorithm}
 * -- 2.2 降低解析与转换成本，而官方的实现使用的是先转换成字符串，然后再转换成 LocalDateTime
 * </pre>
 * 分配策略=[逻辑名] + [分片算法]，分片算法KEY在全局唯一，这种方式显然是不利于缓存优化的，即{@link #doSharding(Collection, RangeShardingValue)}的第一个参数availableTargetNames应该在绑定时已知且稳定，作为实例变量更利于性能。
 *
 * @author ahoo wang
 * @see CosIdIntervalShardingAlgorithm
 * @see CosIdSnowflakeIntervalShardingAlgorithm
 */
public abstract class AbstractIntervalShardingAlgorithm<T extends Comparable<?>> implements StandardShardingAlgorithm<T> {

    public static final String TYPE_PREFIX = CosIdAlgorithm.TYPE_PREFIX + "INTERVAL_";

    public static final String DEFAULT_DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";

    public static final DateTimeFormatter DEFAULT_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern(DEFAULT_DATE_TIME_PATTERN);

    public static final String DATE_TIME_LOWER_KEY = "datetime-lower";

    public static final String DATE_TIME_UPPER_KEY = "datetime-upper";

    public static final String SHARDING_SUFFIX_FORMAT_KEY = "sharding-suffix-pattern";

    public static final String INTERVAL_UNIT_KEY = "datetime-interval-unit";

    public static final String INTERVAL_AMOUNT_KEY = "datetime-interval-amount";

    public static final String ZONE_ID_KEY = "zone-id";

    private ZoneId zoneId = ZoneId.systemDefault();

    private Properties props = new Properties();

    private volatile Sharding<LocalDateTime> sharding;

    @Override
    public Properties getProps() {
        return props;
    }

    @Override
    public void setProps(Properties props) {
        this.props = props;
    }

    protected ZoneId getZoneId() {
        return zoneId;
    }

    @Override
    public void init() {
        if (getProps().containsKey(ZONE_ID_KEY)) {
            zoneId = ZoneId.of(getRequiredValue(ZONE_ID_KEY));
        }
        String logicNamePrefix = getRequiredValue(CosIdAlgorithm.LOGIC_NAME_PREFIX_KEY);
        LocalDateTime effectiveLower = LocalDateTime.parse(getRequiredValue(DATE_TIME_LOWER_KEY), DEFAULT_DATE_TIME_FORMATTER);
        LocalDateTime effectiveUpper = LocalDateTime.parse(getRequiredValue(DATE_TIME_UPPER_KEY), DEFAULT_DATE_TIME_FORMATTER);
        DateTimeFormatter suffixFormatter = DateTimeFormatter.ofPattern(getRequiredValue(SHARDING_SUFFIX_FORMAT_KEY));
        ChronoUnit stepUnit = ChronoUnit.valueOf(getRequiredValue(INTERVAL_UNIT_KEY));
        int stepAmount = Integer.parseInt(getProps().getOrDefault(INTERVAL_AMOUNT_KEY, "1").toString());
        this.sharding = new IntervalTimeline(logicNamePrefix, Range.closed(effectiveLower, effectiveUpper), IntervalStep.of(stepUnit, stepAmount), suffixFormatter);
    }

    protected String getRequiredValue(final String key) {
        return PropertiesUtil.getRequiredValue(getProps(), key);
    }

    @VisibleForTesting
    public Sharding<LocalDateTime> getSharding() {
        return sharding;
    }

    @Override
    public String doSharding(final Collection<String> availableTargetNames, final PreciseShardingValue<T> shardingValue) {
        LocalDateTime shardingTime = convertShardingValue(shardingValue.getValue());
        return this.sharding.sharding(shardingTime);
    }

    @Override
    public Collection<String> doSharding(final Collection<String> availableTargetNames, final RangeShardingValue<T> shardingValue) {
        Range<LocalDateTime> shardingRangeTime = convertRangeShardingValue(shardingValue.getValueRange());
        return this.sharding.sharding(shardingRangeTime);
    }

    /**
     * convert sharding value to {@link LocalDateTime}.
     *
     * @param shardingValue sharding value
     * @return The {@link LocalDateTime} represented by the sharding value
     */
    protected abstract LocalDateTime convertShardingValue(T shardingValue);

    protected Range<LocalDateTime> convertRangeShardingValue(final Range<T> shardingValue) {
        if (Range.all().equals(shardingValue)) {
            return Range.all();
        }
        Object endpointValue = shardingValue.hasLowerBound() ? shardingValue.lowerEndpoint() : shardingValue.upperEndpoint();
        if (endpointValue instanceof LocalDateTime) {
            @SuppressWarnings("unchecked")
            Range<LocalDateTime> targetRange = (Range<LocalDateTime>) shardingValue;
            return targetRange;
        }

        if (shardingValue.hasLowerBound() && shardingValue.hasUpperBound()) {
            LocalDateTime lower = convertShardingValue(shardingValue.lowerEndpoint());
            LocalDateTime upper = convertShardingValue(shardingValue.upperEndpoint());
            return Range.range(lower, shardingValue.lowerBoundType(), upper, shardingValue.upperBoundType());
        }

        if (shardingValue.hasLowerBound()) {
            LocalDateTime lower = convertShardingValue(shardingValue.lowerEndpoint());
            if (BoundType.OPEN.equals(shardingValue.lowerBoundType())) {
                return Range.greaterThan(lower);
            }
            return Range.atLeast(lower);
        }

        LocalDateTime upper = convertShardingValue(shardingValue.upperEndpoint());
        if (BoundType.OPEN.equals(shardingValue.upperBoundType())) {
            return Range.lessThan(upper);
        }
        return Range.atMost(upper);
    }

}
