/*
 * Decompiled with CFR 0.152.
 */
package me.ahoo.cosid.shardingsphere.sharding.interval;

import com.google.common.collect.BoundType;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Range;
import com.google.common.collect.Sets;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
public class IntervalTimeline {
    private final Range<LocalDateTime> effectiveInterval;
    private final Step step;
    private final Interval startInterval;
    private final Interval[] intervals;
    private final String logicName;
    private final DateTimeFormatter suffixFormatter;
    private final Collection<String> allNodes;

    public IntervalTimeline(String logicName, Range<LocalDateTime> effectiveInterval, Step step, DateTimeFormatter suffixFormatter) {
        this.effectiveInterval = effectiveInterval;
        this.step = step;
        this.logicName = logicName;
        this.suffixFormatter = suffixFormatter;
        this.intervals = IntervalTimeline.initIntervals(effectiveInterval, step, logicName, suffixFormatter);
        this.startInterval = this.intervals[0];
        this.allNodes = (Collection)Arrays.stream(this.intervals).map(Interval::getNode).collect(ImmutableSet.toImmutableSet());
    }

    private static Interval[] initIntervals(Range<LocalDateTime> effectiveInterval, Step step, String logicName, DateTimeFormatter suffixFormatter) {
        LocalDateTime lower = step.ofUnit((LocalDateTime)effectiveInterval.lowerEndpoint());
        LocalDateTime upper = step.ofUnit((LocalDateTime)effectiveInterval.upperEndpoint());
        ArrayList<Interval> intervalList = new ArrayList<Interval>();
        while (!lower.isAfter(upper)) {
            String nodeName = logicName + lower.format(suffixFormatter);
            intervalList.add(new Interval(lower, nodeName));
            lower = step.next(lower);
        }
        return intervalList.toArray(new Interval[intervalList.size()]);
    }

    public int size() {
        return this.intervals.length;
    }

    public boolean contains(LocalDateTime time) {
        return this.effectiveInterval.contains((Comparable)time);
    }

    public Interval getStartInterval() {
        return this.startInterval;
    }

    public String getNode(LocalDateTime shardingValue) {
        if (!this.contains(shardingValue)) {
            return null;
        }
        int offset = this.step.unitOffset(this.startInterval.getLower(), shardingValue);
        return this.intervals[offset].getNode();
    }

    public Collection<String> getAllNodes() {
        return this.allNodes;
    }

    public Collection<String> getRangeNode(Range<LocalDateTime> shardingValue) {
        int upperOffset;
        int lowerOffset;
        int maxOffset = this.size() - 1;
        int n = lowerOffset = !shardingValue.hasLowerBound() ? 0 : this.step.unitOffset(this.startInterval.getLower(), (LocalDateTime)shardingValue.lowerEndpoint());
        if (lowerOffset < 0) {
            lowerOffset = 0;
        }
        int n2 = upperOffset = !shardingValue.hasUpperBound() ? maxOffset : this.step.unitOffset(this.startInterval.getLower(), (LocalDateTime)shardingValue.upperEndpoint());
        if (upperOffset > maxOffset) {
            upperOffset = maxOffset;
        }
        if (lowerOffset == 0 && upperOffset == maxOffset) {
            return this.allNodes;
        }
        Interval lastInterval = this.intervals[upperOffset];
        if (lowerOffset == upperOffset) {
            return Collections.singleton(lastInterval.getNode());
        }
        if (shardingValue.hasUpperBound() && BoundType.OPEN.equals((Object)shardingValue.upperBoundType()) && lastInterval.getLower().equals(shardingValue.upperEndpoint())) {
            --upperOffset;
        }
        HashSet nodes = Sets.newHashSetWithExpectedSize((int)(upperOffset - lowerOffset + 1));
        while (lowerOffset <= upperOffset) {
            Interval interval = this.intervals[lowerOffset];
            nodes.add(interval.getNode());
            ++lowerOffset;
        }
        return nodes;
    }

    public static class Step {
        public static final int DEFAULT_AMOUNT = 1;
        private final ChronoUnit unit;
        private final int amount;

        public Step(ChronoUnit unit, int amount) {
            this.unit = unit;
            this.amount = amount;
        }

        public ChronoUnit getUnit() {
            return this.unit;
        }

        public int getAmount() {
            return this.amount;
        }

        public LocalDateTime next(LocalDateTime previous) {
            return previous.plus(this.amount, this.unit);
        }

        public LocalDateTime ofUnit(LocalDateTime time) {
            switch (this.unit) {
                case YEARS: {
                    return LocalDateTime.of(time.getYear(), 1, 1, 0, 0);
                }
                case MONTHS: {
                    return LocalDateTime.of(time.getYear(), time.getMonthValue(), 1, 0, 0);
                }
                case DAYS: {
                    return LocalDateTime.of(time.getYear(), time.getMonthValue(), time.getDayOfMonth(), 0, 0);
                }
                case HOURS: {
                    return LocalDateTime.of(time.getYear(), time.getMonthValue(), time.getDayOfMonth(), time.getHour(), 0);
                }
                case MINUTES: {
                    return LocalDateTime.of(time.getYear(), time.getMonthValue(), time.getDayOfMonth(), time.getHour(), time.getMinute());
                }
                case SECONDS: {
                    return LocalDateTime.of(time.getYear(), time.getMonthValue(), time.getDayOfMonth(), time.getHour(), time.getMinute(), time.getSecond());
                }
            }
            throw new IllegalStateException("Unexpected value: " + this.unit);
        }

        public int unitOffset(LocalDateTime startInterval, LocalDateTime time) {
            long until = startInterval.until(time, this.unit);
            return (int)until / this.amount;
        }

        public static Step of(ChronoUnit unit) {
            return new Step(unit, 1);
        }

        public static Step of(ChronoUnit unit, int amount) {
            return new Step(unit, amount);
        }
    }

    public static class Interval {
        private final LocalDateTime lower;
        private final String node;

        public Interval(LocalDateTime lower, String node) {
            this.lower = lower;
            this.node = node;
        }

        public LocalDateTime getLower() {
            return this.lower;
        }

        public String getNode() {
            return this.node;
        }
    }
}

