package me.ahoo.eventbus.jdbc;

import com.google.common.base.Throwables;
import lombok.var;
import me.ahoo.eventbus.core.repository.*;
import me.ahoo.eventbus.core.repository.entity.PublishEventCompensationEntity;
import me.ahoo.eventbus.core.repository.entity.PublishEventEntity;
import me.ahoo.eventbus.core.utils.Dates;
import me.ahoo.eventbus.core.utils.Jsons;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.support.GeneratedKeyHolder;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;

/**
 * @author ahoo wang
 */
public class JdbcPublishEventRepository implements PublishEventRepository {
    private final NamedParameterJdbcTemplate jdbcTemplate;

    public JdbcPublishEventRepository(NamedParameterJdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    private static final String SQL_INITIALIZED
            = "insert publish_event (event_name, event_data, status,version) values (:event_name, :event_data, :status,:version);";

    @Override
    public PublishIdentity initialize(String eventName, Object eventData) {
        var subscribeIdentity = PublishIdentity.builder()
                .status(PublishStatus.INITIALIZED)
                .version(Version.INITIAL_VALUE)
                .eventName(eventName)
                .build();

        MapSqlParameterSource sqlParams = new MapSqlParameterSource("event_name", eventName);
        String eventDataStr = Jsons.serializeAsString(eventData);
        sqlParams.addValue("event_data", eventDataStr);
        sqlParams.addValue("status", subscribeIdentity.getStatus().getValue());
        sqlParams.addValue("version", subscribeIdentity.getVersion());
        var keyHolder = new GeneratedKeyHolder();
        jdbcTemplate.update(SQL_INITIALIZED, sqlParams, keyHolder);
        subscribeIdentity.setId(keyHolder.getKey().longValue());
        return subscribeIdentity;
    }

    private static final String SQL_MARK_STATUS
            = "update publish_event set status=:status,version=version+1 where id=:id and version=:version;";

    @Override
    public int markStatus(PublishIdentity publishIdentity) {
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("id", publishIdentity.getId());
        sqlParams.addValue("status", publishIdentity.getStatus().getValue());
        sqlParams.addValue("version", publishIdentity.getVersion());

        var affected = jdbcTemplate.update(SQL_MARK_STATUS, sqlParams);
        if (affected == 0) {
            var errMsg = String.format("Publish [%s] mark [%d]@[%d] to status [%s] error."
                    , publishIdentity.getEventName()
                    , publishIdentity.getId()
                    , publishIdentity.getVersion()
                    , publishIdentity.getStatus().name());
            throw new ConcurrentVersionConflictException(errMsg, publishIdentity);
        }
        return affected;
    }

    @Override
    public int markSucceeded(PublishIdentity publishIdentity) {
        publishIdentity.setStatus(PublishStatus.SUCCEEDED);
        return markStatus(publishIdentity);
    }

    private static final String SQL_PUBLISH_FAILED
            = "insert publish_event_failed(publish_event_id, failed_msg) values (:publish_event_id, :failed_msg)";

    /**
     * 1. first insert log
     * 2. last mark status to failed
     *
     * @param publishIdentity
     * @param throwable
     * @return
     */
    @Override
    public int markFailed(PublishIdentity publishIdentity, Throwable throwable) {

        MapSqlParameterSource sqlParams = new MapSqlParameterSource("publish_event_id", publishIdentity.getId());
        var failedMsg = Throwables.getStackTraceAsString(throwable);
        sqlParams.addValue("failed_msg", failedMsg);
        jdbcTemplate.update(SQL_PUBLISH_FAILED, sqlParams);

        publishIdentity.setStatus(PublishStatus.FAILED);
        return markStatus(publishIdentity);
    }

    private static final String SQL_QUERY_FAILED
            = "select id, event_name, event_data, status, published_time, version, create_time from publish_event " +
            "where status<>1 and create_time<:before and version<:max_version order by version asc limit :limit;";

    @Override
    public List<PublishEventEntity> queryFailed(int limit, int before, int maxVersion) {
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("max_version", maxVersion);
        var beforeDate = LocalDateTime.now().minus(before, ChronoUnit.MINUTES);
        sqlParams.addValue("before", beforeDate);
        sqlParams.addValue("limit", limit);
        return jdbcTemplate.query(SQL_QUERY_FAILED, sqlParams, (rs, rowNum) -> {
            var id = rs.getLong("id");
            var eventName = rs.getString("event_name");
            var eventDataStr = rs.getString("event_data");
            var status = rs.getInt("status");
            var publishedTime = rs.getTimestamp("published_time");
            var version = rs.getInt("version");
            var createTime = rs.getTimestamp("create_time");
            var entity = new PublishEventEntity();
            entity.setId(id);
            entity.setEventName(eventName);
            entity.setEventData(eventDataStr);
            entity.setStatus(PublishStatus.valueOf(status));
            entity.setVersion(version);
            entity.setPublishedTime(Dates.of(publishedTime));
            entity.setCreateTime(Dates.of(createTime));
            return entity;
        });
    }

    private static String SQL_COMPENSATE
            = "insert publish_event_compensation (publish_event_id, start_time, taken, failed_msg) " +
            "values (:publish_event_id, :start_time, :taken, :failed_msg);";

    @Override
    public int compensate(PublishEventCompensationEntity publishEventCompensationEntity) {
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("publish_event_id", publishEventCompensationEntity.getPublishEventId());
        sqlParams.addValue("start_time", publishEventCompensationEntity.getStartTime());
        sqlParams.addValue("taken", publishEventCompensationEntity.getTaken());
        sqlParams.addValue("failed_msg", publishEventCompensationEntity.getFailedMsg());
        return jdbcTemplate.update(SQL_COMPENSATE, sqlParams);
    }
}
